/**
 * (c) 2003-2015 MuleSoft, Inc. The software in this package is
 * published under the terms of the CPAL v1.0 license, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.modules.sns;

import com.amazonaws.services.sns.AmazonSNS;
import com.amazonaws.services.sns.model.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.lifecycle.OnException;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.modules.sns.exception.ExceptionDecorator;
import org.mule.modules.sns.model.*;
import org.mule.modules.sns.util.SNSModelFactory;

import java.util.Map;

/**
 * Amazon Simple Notification Service (Amazon SNS) is a web service that makes it easy to set up, operate, and send notifications from the cloud. It provides developers with a
 * highly scalable, flexible, and cost-effective capability to publish messages from an application and immediately deliver them to subscribers or other applications. It is
 * designed to make web-scale computing easier for developers.
 *
 * @author MuleSoft, Inc.
 */

@OnException(handler = ExceptionDecorator.class)
@Connector(name = "sns", schemaVersion = "1.0", friendlyName = "Amazon SNS")
public class AmazonSNSConnector {

    @NotNull
    @org.mule.api.annotations.Config
    private Config config;

    @NotNull
    private AmazonSNS client;

    /**
     * Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:add-permission}
     *
     * @param addPermission
     *            Container for the parameters to the AddPermission operation.
     */
    @Processor
    public void addPermission(@Default("#[payload]") AddPermission addPermission) {
        AddPermissionRequest addPermissionRequest = new AddPermissionRequest(addPermission.getTopicArn(), addPermission.getLabel(), addPermission.getaWSAccountIds(),
                addPermission.getActionNames());
        getClient().addPermission(addPermissionRequest);
    }

    /**
     * The ConfirmSubscription action verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action. If the
     * token is valid, the action creates a new subscription and returns its Amazon Resource Name (ARN). This call requires an AWS signature only when the AuthenticateOnUnsubscribe
     * flag is set to "true".
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:confirm-subscription}
     *
     * @param confirmSubscription
     *            Container for the parameters to the ConfirmSubscription operation.
     * @return ARN of the created subscription.
     */
    @Processor
    public String confirmSubscription(@Default("#[payload]") ConfirmSubscription confirmSubscription) {
        ConfirmSubscriptionRequest confirmSubscriptionRequest = new ConfirmSubscriptionRequest(confirmSubscription.getTopicArn(), confirmSubscription.getToken(),
                confirmSubscription.getAuthenticateOnUnsubscribe());
        return getClient().confirmSubscription(confirmSubscriptionRequest).getSubscriptionArn();
    }

    /**
     * The CreateTopic action creates a topic to which notifications can be published. Users can create at most 100 topics
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:create-topic}
     *
     * @param topicName
     *            The name of the topic you want to create.
     * @return The Amazon Resource Name (ARN) assigned to the created topic.
     */
    @Processor
    public String createTopic(@NotNull String topicName) {
        CreateTopicRequest createTopicRequest = new CreateTopicRequest(topicName);
        return getClient().createTopic(createTopicRequest).getTopicArn();
    }

    /**
     * The DeleteTopic action deletes a topic and all its subscriptions. Deleting a topic might prevent some messages previously sent to the topic from being delivered to
     * subscribers. This action is idempotent, so deleting a topic that does not exist does not result in an error.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:delete-topic}
     *
     * @param topicArn
     *            The ARN of the topic you want to delete.
     */
    @Processor
    public void deleteTopic(@NotNull String topicArn) {
        DeleteTopicRequest deleteTopicRequest = new DeleteTopicRequest(topicArn);
        getClient().deleteTopic(deleteTopicRequest);
    }

    /**
     * The GetSubscriptionAttributes action returns all of the properties of a subscription.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:get-subscription-attributes}
     *
     * @param subscriptionArn
     *            The ARN of the subscription whose properties you want to get
     * @return A map of the subscription's attributes.
     */
    @Processor
    public Map<String, String> getSubscriptionAttributes(@NotNull String subscriptionArn) {
        GetSubscriptionAttributesRequest getSubscriptionAttributesRequest = new GetSubscriptionAttributesRequest(subscriptionArn);
        return getClient().getSubscriptionAttributes(getSubscriptionAttributesRequest).getAttributes();
    }

    /**
     * The GetTopicAttributes action returns all of the properties of a topic. Topic properties returned might differ based on the authorization of the user.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:get-topic-attributes}
     *
     * @param topicArn
     *            The ARN of the topic whose properties you want to get.
     * @return A map of the topic's attributes.
     */
    @Processor
    public Map<String, String> getTopicAttributes(@NotNull String topicArn) {
        GetTopicAttributesRequest getTopicAttributesRequest = new GetTopicAttributesRequest(topicArn);
        return getClient().getTopicAttributes(getTopicAttributesRequest).getAttributes();
    }

    /**
     * The ListSubscriptions action returns a list of the requester's subscriptions. Each call returns a limited list of subscriptions, up to 100. If there are more subscriptions,
     * a NextToken is also returned. Use the NextToken parameter in a new ListSubscriptions call to get further results.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:list-subscriptions}
     *
     * @param nextToken
     *            Token returned by the previous ListSubscriptions request
     * @return Returns a list of the requester's subscriptions.
     */
    @Processor
    public ListSubscriptions listSubscriptions(@Optional @Nullable String nextToken) {
        ListSubscriptionsRequest listSubscriptionsRequest = new ListSubscriptionsRequest(nextToken);
        ListSubscriptionsResult listSubscriptionsResult = getClient().listSubscriptions(listSubscriptionsRequest);
        return SNSModelFactory.getListSubscriptions(listSubscriptionsResult);
    }

    /**
     * The SubscriptionsByTopic action returns a list of the subscriptions to a specific topic. Each call returns a limited list of subscriptions, up to 100. If there are more
     * subscriptions, a NextToken is also returned. Use the NextToken parameter in a new SubscriptionsByTopic call to get further results.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:list-subscriptions-by-topic}
     *
     * @param subscriptionsByTopic
     *            Container for the parameters to the ListSubscriptionsByTopic operation.
     * @return Returns a list of the subscriptions to a specific topic.
     */
    @Processor
    public ListSubscriptionsByTopic listSubscriptionsByTopic(@Default("#[payload]") SubscriptionsByTopic subscriptionsByTopic) {
        ListSubscriptionsByTopicRequest listSubscriptionsByTopicRequest = new ListSubscriptionsByTopicRequest(subscriptionsByTopic.getTopicArn(),
                subscriptionsByTopic.getNextToken());
        return SNSModelFactory.getListSubscriptionsByTopic(getClient().listSubscriptionsByTopic(listSubscriptionsByTopicRequest));
    }

    /**
     * The ListTopics action returns a list of the requester's topics. Each call returns a limited list of topics, up to 100. If there are more topics, a NextToken is also
     * returned. Use the NextToken parameter in a new ListTopics call to get further results.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:list-topics}
     *
     * @param nextTopic
     *            Token returned by the previous ListTopics request.
     * @return Returns a list of the requester's topics.
     */
    @Processor
    public ListTopics listTopics(@Optional @Nullable String nextTopic) {
        ListTopicsRequest listTopicsRequest = new ListTopicsRequest(nextTopic);
        return SNSModelFactory.getListTopics(getClient().listTopics(listTopicsRequest));
    }

    /**
     * The Publish action sends a message to all of a topic's subscribed endpoints. When a messageId is returned, the message has been saved and Amazon SNS will attempt to deliver
     * it to the topic's subscribers shortly. The format of the outgoing message to each subscribed endpoint depends on the notification protocol selected.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_Publish.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:publish}
     *
     * @param publish
     *            Container for the parameters to the publish operation.
     * @return Unique identifier assigned to the published message.
     */
    @Processor
    public String publish(@Default("#[payload]") Publish publish) {
        PublishRequest publishRequest = new PublishRequest(publish.getTopicArn(), publish.getMessage(), publish.getSubject());
        publishRequest.setMessageStructure(publish.getMessageStructure());
        publishRequest.setTargetArn(publish.getTargetArn());
        publishRequest.setMessageAttributes(SNSModelFactory.getMessageAttributes(publish.getMessageAttributes()));

        return getClient().publish(publishRequest).getMessageId();
    }

    /**
     * The RemovePermission action removes a statement from a topic's access control policy.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:remove-permission}
     * 
     * @param removePermission
     *            Container for the parameters to the RemovePermission operation.
     */
    @Processor
    public void removePermission(@Default("#[payload]") RemovePermission removePermission) {
        RemovePermissionRequest removePermissionRequest = new RemovePermissionRequest(removePermission.getTopicArn(), removePermission.getLabel());
        getClient().removePermission(removePermissionRequest);
    }

    /**
     * The SubscriptionAttributes action allows a subscription owner to set an attribute of the topic to a new value.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:set-subscription-attributes}
     * 
     * @param subscriptionAttributes
     *            Container for the parameters to the SetSubscriptionAttributes operation.
     */
    @Processor
    public void setSubscriptionAttributes(@Default("#[payload]") SubscriptionAttributes subscriptionAttributes) {
        SetSubscriptionAttributesRequest setSubscriptionAttributesRequest = new SetSubscriptionAttributesRequest(subscriptionAttributes.getSubscriptionArn(),
                subscriptionAttributes.getAttributeName(), subscriptionAttributes.getAttributeValue());
        getClient().setSubscriptionAttributes(setSubscriptionAttributesRequest);
    }

    /**
     * The TopicAttributes action allows a topic owner to set an attribute of the topic to a new value.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:set-topic-attributes}
     * 
     * @param topicAttributes
     *            Container for the parameters to the SetTopicAttributes operation.
     */
    @Processor
    public void setTopicAttributes(@Default("#[payload]") TopicAttributes topicAttributes) {
        SetTopicAttributesRequest setTopicAttributesRequest = new SetTopicAttributesRequest(topicAttributes.getTopicArn(), topicAttributes.getAttributeName(),
                topicAttributes.getAttributeValue());
        getClient().setTopicAttributes(setTopicAttributesRequest);
    }

    /**
     * The Subscribe action prepares to subscribe an endpoint by sending the endpoint a confirmation message. To actually create a subscription, the endpoint owner must call the
     * ConfirmSubscription action with the token from the confirmation message. Confirmation tokens are valid for three days.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:subscribe}
     * 
     * @param subscribe
     *            Container for the parameters to the Subscribe operation.
     * @return The ARN of the subscription, if the service was able to create a subscription immediately (without requiring endpoint owner confirmation).
     */
    @Processor
    public String subscribe(@Default("#[payload]") Subscribe subscribe) {
        SubscribeRequest subscribeRequest = new SubscribeRequest(subscribe.getTopicArn(), subscribe.getProtocol(), subscribe.getEndpoint());
        return getClient().subscribe(subscribeRequest).getSubscriptionArn();
    }

    /**
     * The Unsubscribe action deletes a subscription. If the subscription requires authentication for deletion, only the owner of the subscription or the its topic's owner can
     * unsubscribe, and an AWS signature is required. If the Unsubscribe call does not require authentication and the requester is not the subscription owner, a final cancellation
     * message is delivered to the endpoint, so that the endpoint owner can easily resubscribe to the topic if the Unsubscribe request was unintended.
     * <p>
     * <a href=http://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html>API reference</a>
     * </p>
     *
     * {@sample.xml ../../../doc/amazon-sns-connector.xml.sample sns:unsubscribe}
     *
     * @param subscriptionArn
     *            The ARN of the subscription to be deleted.
     */
    @Processor
    public void unsubscribe(@NotNull String subscriptionArn) {
        UnsubscribeRequest unsubscribeRequest = new UnsubscribeRequest(subscriptionArn);
        getClient().unsubscribe(unsubscribeRequest);
    }

    @NotNull
    public AmazonSNS getClient() {
        return client;
    }

    public void setClient(@NotNull AmazonSNS client) {
        this.client = client;
    }

    @NotNull
    public Config getConfig() {
        return config;
    }

    public void setConfig(@NotNull Config config) {
        this.config = config;
        setClient(config.getClient());
    }
}
