/**
 * (c) 2003-2015 MuleSoft, Inc. The software in this package is
 * published under the terms of the CPAL v1.0 license, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.modules.sns;

import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.regions.Region;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.sns.AmazonSNS;
import com.amazonaws.services.sns.AmazonSNSClient;
import org.apache.commons.lang.StringUtils;
import org.mule.api.ConnectionException;
import org.mule.api.ConnectionExceptionCode;
import org.mule.api.annotations.*;
import org.mule.api.annotations.components.ConnectionManagement;
import org.mule.api.annotations.display.FriendlyName;
import org.mule.api.annotations.display.Placement;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.api.annotations.param.Optional;
import org.mule.modules.sns.model.RegionEndpoint;

@ConnectionManagement(friendlyName = "Configuration")
public class Config {

    private AmazonSNS client;

    /**
     * Set the topic region endpoint
     */
    @Optional
    @Configurable
    @Placement(group = "Optional Parameters", order = 1)
    @FriendlyName("Region Endpoint")
    private RegionEndpoint region;

    /**
     * Arn of the topic to test. This is used only to test the connectivity and validate credentials with Amazon SNS.
     */
    @Optional
    @Configurable
    @Placement(group = "Optional Parameters", order = 2)
    @FriendlyName("Topic Arn (Only to test connectivity)")
    private String testTopicArn;

    /**
     * Login to Amazon SNS
     *
     * @param accessKey
     *            The access key provided by Amazon
     * @param secretKey
     *            The secrete key provided by Amazon
     * @throws ConnectionException
     */
    @Connect
    public void connect(@ConnectionKey String accessKey, String secretKey) throws ConnectionException {

        if (StringUtils.isBlank(accessKey) || StringUtils.isBlank(secretKey)) {
            throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, "001", "Access Key or Secret Key is blank");
        }

        client = new AmazonSNSClient(new BasicAWSCredentials(accessKey, secretKey));

        if (region != null) {
            client.setEndpoint(region.value());
        }
    }

    /**
     * Test connectivity to Amazon SNS
     *
     * @param accessKey
     *            The access key provided by Amazon
     * @param secretKey
     *            The secrete key provided by Amazon
     * @throws ConnectionException
     */
    @TestConnectivity
    public void testConnect(@ConnectionKey String accessKey, String secretKey) throws ConnectionException {
        if (StringUtils.isBlank(testTopicArn)) {
            throw new ConnectionException(ConnectionExceptionCode.CANNOT_REACH, "002", "Specify the ARN of the topic to test connectivity.");
        }

        this.connect(accessKey, secretKey);

        if (region == null) {
            client.setRegion(Region.getRegion(Regions.fromName(extractRegion(testTopicArn))));
        }

        try {
            client.getTopicAttributes(testTopicArn);
        } catch (Exception e) {
            throw new ConnectionException(ConnectionExceptionCode.UNKNOWN, null,
                    "Invalid Topic Arn or Topic belongs to different region than mentioned in Region Endpoint attribute.\n ", e);
        }
    }

    /**
     * Disconnect
     */
    @Disconnect
    public void disconnect() {
        setClient(null);
    }

    /**
     * Check if the connector's instance is connected
     */
    @ValidateConnection
    public boolean isConnected() {
        return getClient() != null;
    }

    /**
     * Connection identifier
     */
    @ConnectionIdentifier
    public String connectionId() {
        return "amazon-sns-";
    }

    private String extractRegion(String testTopicArn) {
        String[] strings = testTopicArn.split(":");
        if (strings.length > 3) {
            return strings[3];
        }
        return "us-east-1";
    }

    public AmazonSNS getClient() {
        return client;
    }

    public void setClient(AmazonSNS client) {
        this.client = client;
    }

    public RegionEndpoint getRegion() {
        return region;
    }

    public void setRegion(RegionEndpoint region) {
        this.region = region;
    }

    public String getTestTopicArn() {
        return testTopicArn;
    }

    public void setTestTopicArn(String testTopicArn) {
        this.testTopicArn = testTopicArn;
    }
}
