package org.mule.datasense.impl.phases.typing.resolver.transform;

import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.toFile;
import static org.mule.datasense.impl.model.reporting.NotificationMessages.MSG_RESOURCE_NOT_FOUND;
import static org.mule.datasense.impl.phases.builder.MuleAstParser.MULE_EE_CORE;
import static org.mule.datasense.impl.phases.builder.MuleAstParser.MULE_EE_CORE_PREFIX;

import org.mule.datasense.impl.model.ast.AstNodeLocation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.event.MuleEventExprBuilder;
import org.mule.datasense.impl.model.event.SimpleExprBuilder;
import org.mule.datasense.impl.model.event.ValueExprBuilder;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.ast.api.ComponentAst;

import java.net.URI;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

public class TransformParser {

  static final String MESSAGE = "message";
  static final String VARIABLES = "variables";
  static final String SET_PAYLOAD = "setPayload";
  static final String SET_ATTRIBUTES = "setAttributes";
  static final ComponentIdentifier SET_VARIABLE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE_PREFIX, MULE_EE_CORE, "set-variable");
  static final String ATTR_VARIABLE_NAME = "variableName";
  static final String ATTR_RESOURCE = "resource";
  static final String SCRIPT_PARAMETER = "script";

  private static class TransformParserContext {

    private final Function<String, Optional<URI>> resourceResolver;
    private final AstNotification astNotification;
    private final AstNodeLocation astNodeLocation;

    public TransformParserContext(Function<String, Optional<URI>> resourceResolver,
                                  AstNotification astNotification, AstNodeLocation astNodeLocation) {
      this.resourceResolver = resourceResolver;
      this.astNotification = astNotification;
      this.astNodeLocation = astNodeLocation;
    }

    public Optional<Function<String, Optional<URI>>> getResourceResolver() {
      return ofNullable(resourceResolver);
    }

    public Optional<AstNotification> getAstNotification() {
      return ofNullable(astNotification);
    }

    public Optional<AstNodeLocation> getAstNodeLocation() {
      return ofNullable(astNodeLocation);
    }
  }

  private Optional<String> findResourceExpression(String resource, TransformParserContext transformParserContext) {
    return transformParserContext.getResourceResolver().map(resourceResolver -> {
      return resourceResolver.apply(resource).map(uri -> {
        try {
          return readFileToString(toFile(uri.toURL()));
        } catch (Exception e) {
          transformParserContext.getAstNotification()
              .ifPresent(astNotification -> astNotification.reportError(transformParserContext.getAstNodeLocation().orElse(null),
                                                                        MSG_RESOURCE_NOT_FOUND(resource)));
          return null;
        }
      });
    }).orElse(empty());
  }

  private Optional<String> findExpression(ComponentAst componentModel, TransformParserContext transformParserContext) {
    final Either<String, Optional<String>> resolvedResource = componentModel.getParameter(ATTR_RESOURCE).getValue()
        .mapRight(resource -> findResourceExpression((String) resource, transformParserContext));

    if (resolvedResource.isRight()) {
      return resolvedResource.getRight();
    } else {
      return ofNullable((String) componentModel.getParameter(SCRIPT_PARAMETER).getValue().getRight());
    }
  }

  public void parseVariables(List<ComponentAst> componentModels, MuleEventExprBuilder muleEventExprBuilder,
                             TransformParserContext transformParserContext) {
    componentModels.stream()
        .forEach(childComponentModel -> {
          ComponentIdentifier componentIdentifier = childComponentModel.getIdentifier();
          // TODO MULE-17710 Use the parameters instead of querying the dsl children
          if (SET_VARIABLE.equals(componentIdentifier)) {
            childComponentModel.getParameter(ATTR_VARIABLE_NAME).getValue()
                .applyRight(name -> findExpression(childComponentModel, transformParserContext)
                    .ifPresent(expression -> muleEventExprBuilder
                        .variable((String) name, v -> v.value(new ValueExprBuilder(new SimpleExprBuilder(expression))))));
          }
        });
  }

  public MuleEventExprBuilder parse(ComponentAst transformComponentModel, TransformParserContext transformParserContext) {
    MuleEventExprBuilder muleEventExprBuilder = new MuleEventExprBuilder();

    transformComponentModel.getParameter(SET_PAYLOAD).getValue()
        .applyRight(setPayload -> findExpression((ComponentAst) setPayload, transformParserContext)
            .ifPresent(expression -> muleEventExprBuilder
                .message(m -> m.value(m1 -> m1.payload(new SimpleExprBuilder(expression))))));

    transformComponentModel.getParameter(SET_ATTRIBUTES).getValue()
        .applyRight(setAttributes -> findExpression((ComponentAst) setAttributes, transformParserContext)
            .ifPresent(expression -> muleEventExprBuilder
                .message(m -> m.value(m1 -> m1.attributes(new SimpleExprBuilder(expression))))));

    transformComponentModel.getParameter(VARIABLES).getValue()
        .applyRight(variables -> parseVariables((List<ComponentAst>) variables, muleEventExprBuilder, transformParserContext));

    return muleEventExprBuilder;
  }

  public MuleEventExprBuilder parse(ComponentAst transformComponentModel, Function<String, Optional<URI>> resourceResolver,
                                    AstNotification astNotification, AstNodeLocation astNodeLocation) {
    return parse(transformComponentModel, new TransformParserContext(resourceResolver, astNotification, astNodeLocation));
  }

  public MuleEventExprBuilder parse(ComponentAst transformComponentModel) {
    return parse(transformComponentModel, null, null, null);
  }

}
