package org.mule.datasense.impl.util;

import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.IntersectionTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TupleTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;

import java.util.List;
import java.util.regex.Pattern;

import static org.mule.datasense.impl.util.Parsers.begin;
import static org.mule.datasense.impl.util.Parsers.end;
import static org.mule.datasense.impl.util.Parsers.regex;
import static org.mule.datasense.impl.util.Parsers.skip;

public class MetadataTypeReader {
  //    private static TypeAnnotationNameClassMapping typeAnnotationNameClassMapping = new TypeAnnotationNameClassMapping();

  private static final String NUMBER_TYPE = "Number";
  private static final String STRING_TYPE = "String";
  private static final String BOOLEAN_TYPE = "Boolean";
  private static final String ANY_TYPE = "Any";
  private static final String BINARY_TYPE = "Binary";
  private static final String DATETIME_TYPE = "DateTime";
  private static final String DATE_TYPE = "Date";
  private static final String LOCALDATETIME_TYPE = "LocalDateTime";
  private static final String LOCALTIME_TYPE = "LocalTime";
  private static final String NOTHING_TYPE = "Nothing";
  private static final String NULL_TYPE = "Null";
  private static final String PERIOD_TYPE = "Period";
  private static final String TIMEZONE_TYPE = "TimeZone";

  static Parser<?> token(String keyword) {
    return regex(Pattern.quote(keyword) + "[\\s\\r\\n]*");
  }

  static Parser<TypeBuilder> parseNumber = token(NUMBER_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).numberType());
  static Parser<TypeBuilder> parseString = token(STRING_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).stringType());
  static Parser<TypeBuilder> parseBoolean = token(BOOLEAN_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).booleanType());
  static Parser<TypeBuilder> parseAny = token(ANY_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).anyType());
  static Parser<TypeBuilder> parseBinary = token(BINARY_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).binaryType());
  static Parser<TypeBuilder> parseDateTime =
      token(DATETIME_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).dateTimeType());
  static Parser<TypeBuilder> parseDate = token(DATE_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).dateType());
  static Parser<TypeBuilder> parseLocalDateTime =
      token(LOCALDATETIME_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).localDateTimeType());
  static Parser<TypeBuilder> parseLocalTime =
      token(LOCALTIME_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).localTimeType());
  static Parser<TypeBuilder> parseNothing = token(NOTHING_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).nothingType());
  static Parser<TypeBuilder> parseNull = token(NULL_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).nullType());
  static Parser<TypeBuilder> parsePeriod = token(PERIOD_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).periodType());
  static Parser<TypeBuilder> parseTimeZone =
      token(TIMEZONE_TYPE).map(o -> new BaseTypeBuilder(MetadataFormat.JAVA).timeZoneType());

  static Parser<?> parseBs = regex(Pattern.compile("[\\s\\r\\n]*"));

  static Parser<String> parseWord = regex(Pattern.compile("([0-9a-zA-Z])+")).map(matchResult -> matchResult.group(0));


  static Parser<TypeBuilder> parseHeader =
      skip(
           token("%type").then(token("_:")))
               .then(parseWord.map(format -> (TypeBuilder) new BaseTypeBuilder(new MetadataFormat(format, format, null))));

  static Parser<TypeBuilder> parseBasicType =
      parseString.or(parseBoolean).or(parseNumber).or(parseAny).or(parseBinary).or(parseDateTime).or(parseDate)
          .or(parseLocalDateTime).or(parseLocalTime).or(parseNothing).or(parseNull).or(parsePeriod).or(parseTimeZone);


  static Parser<TypeBuilder> parseArray =
      skip(token("[")).then(() -> MetadataTypeReader.parseType).skip(token("]")).map(typeBuilder -> {
        final ArrayTypeBuilder arrayTypeBuilder = BaseTypeBuilder.create(MetadataFormat.JAVA).arrayType();
        return arrayTypeBuilder.of(typeBuilder);
      });

  static Parser<TypeBuilder> parseUnion =
      skip(begin()).then(() -> MetadataTypeReader.parseTypeWithoutUnionOrIntersection)
          .twoOrMore(token("|")).map(typeBuilders -> {
            final UnionTypeBuilder unionTypeBuilder = BaseTypeBuilder.create(MetadataFormat.JAVA).unionType();
            typeBuilders.forEach(unionTypeBuilder::of);
            return unionTypeBuilder;
          });

  static Parser<TypeBuilder> parseIntersection =
      skip(begin()).then(() -> MetadataTypeReader.parseTypeWithoutUnionOrIntersection)
          .twoOrMore(token("&")).map(typeBuilders -> {
            final IntersectionTypeBuilder intersectionTypeBuilder =
                BaseTypeBuilder.create(MetadataFormat.JAVA).intersectionType();
            typeBuilders.forEach(intersectionTypeBuilder::of);
            return intersectionTypeBuilder;
          });

  static Parser<TypeBuilder> parseParenthesis =
      skip(token("(")).then(() -> MetadataTypeReader.parseType).skip(token(")"));


  static Parser<String> parseAnnotationKey = parseWord;

  static Parser<String> parseAnnotationValue = parseWord;

  static Parser<Parsers.Pair<String, String>> parseAnnotationProperty =
      parseAnnotationKey.skip(token(":")).then(parseAnnotationValue);

  static Parser<List<Parsers.Pair<String, String>>> parseAnnotationProperties = parseAnnotationProperty.oneOrMore(token(","));

  static Parser<TypeAnnotation> parseAnnotation =
      skip(token("@")).then(parseWord).skip(token("(")).then(parseAnnotationProperties).skip(token(")")).map(stringListPair -> {
        final String annotation = stringListPair.first;
        final List<Parsers.Pair<String, String>> annotationProperties = stringListPair.second;

        /*
            final Class<? extends TypeAnnotation> aClass = typeAnnotationNameClassMapping.getNameClassMapping().get(annotation);
            System.out.println("aClass: " + aClass);
        */
        return new DefaultValueAnnotation("");
      });

  static Parser<String> parseKey = parseWord;

  static Parser<TypeBuilder> parseValue = skip(begin()).then(() -> MetadataTypeReader.parseType);

  static Parser<Parsers.Pair<String, TypeBuilder>> parseKeyValue = parseKey.skip(token(":")).then(parseValue);

  static Parser<List<Parsers.Pair<String, TypeBuilder>>> parseKeyValues = parseKeyValue.oneOrMore(token(","));

  static Parser<TypeBuilder> parseObject =
      skip(token("{")).skip(parseAnnotation.optional()).then(() -> MetadataTypeReader.parseKeyValues).skip(token("}"))
          .map(pairs -> {
            final ObjectTypeBuilder objectTypeBuilder = BaseTypeBuilder.create(MetadataFormat.JAVA).objectType();

            pairs.forEach(stringTypeBuilderPair -> {
              final String key = stringTypeBuilderPair.first;
              final TypeBuilder typeBuilder = stringTypeBuilderPair.second;
              objectTypeBuilder.addField().key(key).value(typeBuilder);
            });
            return objectTypeBuilder;
          });

  static Parser<TypeBuilder> parseTuple =
      skip(token("<")).then(() -> MetadataTypeReader.parseType.oneOrMore(token(","))).skip(token(">")).map(typeBuilders -> {
        final TupleTypeBuilder tupleTypeBuilder = BaseTypeBuilder.create(MetadataFormat.JAVA).tupleType();
        typeBuilders.forEach(tupleTypeBuilder::of);
        return tupleTypeBuilder;
      });


  static Parser<TypeBuilder> parseType =
      parseUnion.or(parseIntersection).or(parseParenthesis).or(parseArray).or(parseBasicType);

  static Parser<TypeBuilder> parseTypeWithoutUnionOrIntersection =
      parseArray.or(parseParenthesis).or(parseBasicType);

  static Parser<TypeBuilder> parseMetadataTypeDefinition =
      parseHeader.skip(parseBs.then(token("=").then(parseBs))).then(parseType)
          .map(typeBuilderTypeBuilderPair -> typeBuilderTypeBuilderPair.second);

  public static Parser<TypeBuilder> parse = skip(begin()).then(parseMetadataTypeDefinition).skip(end());

}
