package org.mule.datasense.impl.phases.typing;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.MuleApplicationAnnotation;
import org.mule.datasense.impl.model.annotations.MuleFlowAnnotation;
import org.mule.datasense.impl.model.annotations.ThrowsErrorsTypeAnnotation;
import org.mule.datasense.impl.model.annotations.TypeResolverAnnotation;
import org.mule.datasense.impl.model.annotations.TypesResolvedAnnotation;
import org.mule.datasense.impl.model.ast.AstNode;
import org.mule.datasense.impl.model.ast.AstNodeVisitor;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.datasense.impl.model.ast.MuleFlowNode;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolverRegistry;
import org.mule.datasense.impl.phases.typing.resolver.errorhandling.ErrorHandlingContext;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;

import java.util.Optional;

import com.google.common.base.Preconditions;

;

public class TypingMuleAstVisitor implements AstNodeVisitor<TypingMuleAstVisitorContext> {

  private final TypeResolverRegistry typeResolverRegistry;
  private final ExpressionLanguageMetadataService expressionLanguageMetadataService;

  public TypingMuleAstVisitor(TypeResolverRegistry typeResolverRegistry,
                              ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    Preconditions.checkNotNull(typeResolverRegistry);

    this.typeResolverRegistry = typeResolverRegistry;
    this.expressionLanguageMetadataService = expressionLanguageMetadataService;
  }

  @Override
  public Object visit(MuleApplicationNode muleApplicationNode, TypingMuleAstVisitorContext visitorContext) {
    visitorContext.logger().enter(muleApplicationNode.getName());

    visitorContext.annotate(new MuleApplicationAnnotation(muleApplicationNode));
    muleApplicationNode.getMuleFlowNodes().forEach(muleFlowNode -> {
      muleFlowNode.accept(this, visitorContext);
    });
    visitorContext.deannotate(MuleApplicationAnnotation.class);

    visitorContext.logger().exit(muleApplicationNode.getName());
    return null;
  }

  @Override
  public Object visit(MuleFlowNode muleFlowNode, TypingMuleAstVisitorContext visitorContext) {
    final Optional<TypesResolvedAnnotation> typesResolvedAnnotation = muleFlowNode.getAnnotation(TypesResolvedAnnotation.class);
    if (typesResolvedAnnotation.isPresent()) {
      if (!typesResolvedAnnotation.get().getResult().isPresent()) {
        visitorContext.getAstNotification().reportError(muleFlowNode.getAstNodeLocation(),
                                                        NotificationMessages.MSG_RECURSIVE_CALL_DETECTED(muleFlowNode.getName()));
      }
      return typesResolvedAnnotation.get().getResult().orElse(new EventType());
    }
    muleFlowNode.annotate(new TypesResolvedAnnotation());
    visitorContext.logger().enter(muleFlowNode.getName());
    visitorContext.annotate(new MuleFlowAnnotation(muleFlowNode));
    visitorContext.getErrorHandlingEnvironment()
        .pushContext("muleFlowNode-" + muleFlowNode.getComponentModel().getComponentId().orElse(null));
    EventType result = (EventType) muleFlowNode.getRootMessageProcessorNode()
        .map(messageProcessorNode -> messageProcessorNode.accept(this, visitorContext)).orElse(new EventType());
    if (!muleFlowNode.isAnnotatedWith(DefinesTypeAnnotation.class)) {
      // infer output from propagation
      muleFlowNode.annotate(new DefinesTypeAnnotation(result));
    }
    visitorContext.deannotate(MuleFlowAnnotation.class);
    visitorContext.logger().exit(muleFlowNode.getName());
    muleFlowNode.reannotate(new TypesResolvedAnnotation(result));
    final ErrorHandlingContext errorHandlingContext = visitorContext.getErrorHandlingEnvironment().getContext();
    muleFlowNode.annotate(new ThrowsErrorsTypeAnnotation(errorHandlingContext.getUnhandledErrors()));
    visitorContext.getErrorHandlingEnvironment().popContext();
    return result;
  }

  @Override
  public Object visit(MessageProcessorNode messageProcessorNode, TypingMuleAstVisitorContext typingMuleAstVisitorContext) {
    typingMuleAstVisitorContext.logger().enter(messageProcessorNode.getComponentIdentifier().toString());
    EventType result = typingMuleAstVisitorContext.getTypeResolverRegistry()
        .get(messageProcessorNode.getAnnotation(TypeResolverAnnotation.class)
            .map(TypeResolverAnnotation::getTypeResolver)
            .orElse(null))
        .map(muleAstFunctionTypeResolver -> muleAstFunctionTypeResolver
            .resolveTypes(messageProcessorNode, this, typingMuleAstVisitorContext))
        .orElse(new EventType());
    typingMuleAstVisitorContext.logger().exit(messageProcessorNode.getComponentIdentifier().toString());
    return result;
  }

  public EventType resolveType(AstNode astNode, EventType inputEventType, AstNotification astNotification) {
    return resolveType(astNode, inputEventType, new TypingMuleAstVisitorContext(typeResolverRegistry, astNotification,
                                                                                expressionLanguageMetadataService));
  }

  public EventType resolveType(AstNode astNode, EventType inputEventType, AstNotification astNotification,
                               MuleApplicationNode muleApplicationNode) {
    final TypingMuleAstVisitorContext typingMuleAstVisitorContext =
        new TypingMuleAstVisitorContext(typeResolverRegistry, astNotification, expressionLanguageMetadataService);
    typingMuleAstVisitorContext.annotate(new MuleApplicationAnnotation(muleApplicationNode));
    return resolveType(astNode, inputEventType, typingMuleAstVisitorContext);
  }

  public EventType resolveType(AstNode astNode, EventType inputEvent, TypingMuleAstVisitorContext typingMuleAstVisitorContext) {
    Preconditions.checkNotNull(typingMuleAstVisitorContext);
    if (astNode == null) {
      return new EventType();
    }

    typingMuleAstVisitorContext.getTypingEnvironment().push(new MessageProcessorScope(inputEvent));
    EventType result = (EventType) astNode.accept(this, typingMuleAstVisitorContext);
    typingMuleAstVisitorContext.getTypingEnvironment().pop();
    return result;
  }
}
