/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.datasense.impl.tooling;

import static org.mule.datasense.impl.DataSenseProviderResolver.isDynamicDataSenseSupportedFor;

import org.mule.datasense.impl.DataSenseProviderResolver;
import org.mule.datasense.impl.model.annotations.HasDynamicMetadataAnnotation;
import org.mule.datasense.impl.model.annotations.OperationCallAnnotation;
import org.mule.datasense.impl.model.annotations.OperationModelMetadataKeyEnricher;
import org.mule.datasense.impl.model.annotations.SourceModelMetadataKeyEnricher;
import org.mule.datasense.impl.model.annotations.metadata.OperationModelAnnotation;
import org.mule.datasense.impl.model.annotations.metadata.SourceModelAnnotation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;

import java.util.Optional;

/**
 *
 */
public class MetadataQuery implements TypedApplicationQuery<MetadataQueryResult> {

  private final Location location;
  private final OperationModelMetadataKeyEnricher operationModelMetadataKeyEnricher;
  private final SourceModelMetadataKeyEnricher sourceModelMetadataKeyEnricher;

  /**
   *
   * @param location
   */
  public MetadataQuery(Location location) {
    this.location = location;
    this.operationModelMetadataKeyEnricher = new OperationModelMetadataKeyEnricher();
    this.sourceModelMetadataKeyEnricher = new SourceModelMetadataKeyEnricher();
  }

  private OperationModel enrichOperationModel(MessageProcessorNode messageProcessorNode, OperationModel operationModel,
                                              DataSenseProviderResolver dataSenseProviderResolver,
                                              AstNotification astNotification) {
    return dataSenseProviderResolver.getDataSenseProvider().getDataSenseMetadataProvider().map(dataSenseMetadataProvider -> {
      if (messageProcessorNode.isAnnotatedWith(HasDynamicMetadataAnnotation.class)
          && isDynamicDataSenseSupportedFor(messageProcessorNode)) {
        return operationModelMetadataKeyEnricher
            .enrich(operationModel, messageProcessorNode.getComponentModel(),
                    messageProcessorNode.getAnnotation(OperationCallAnnotation.class)
                        .map(OperationCallAnnotation::getOperationCall)
                        .orElse(null),
                    location, dataSenseMetadataProvider, astNotification,
                    dataSenseProviderResolver.getDataSenseProvider().getDataSenseConfiguration());
      } else {
        return operationModel;
      }
    }).orElse(operationModel);
  }

  private SourceModel enrichSourceModel(MessageProcessorNode messageProcessorNode, SourceModel sourceModel,
                                        DataSenseProviderResolver dataSenseProviderResolver, AstNotification astNotification) {
    return dataSenseProviderResolver.getDataSenseProvider().getDataSenseMetadataProvider().map(dataSenseMetadataProvider -> {
      if (messageProcessorNode.isAnnotatedWith(HasDynamicMetadataAnnotation.class)
          && isDynamicDataSenseSupportedFor(messageProcessorNode)) {
        return sourceModelMetadataKeyEnricher
            .enrich(sourceModel, messageProcessorNode.getComponentModel(),
                    messageProcessorNode
                        .getAnnotation(OperationCallAnnotation.class).map(OperationCallAnnotation::getOperationCall)
                        .orElse(null),
                    location, dataSenseMetadataProvider, astNotification,
                    dataSenseProviderResolver.getDataSenseProvider().getDataSenseConfiguration());
      } else {
        return sourceModel;
      }
    }).orElse(sourceModel);
  }

  /**
   *
   * @param muleApplicationNode
   * @param dataSenseProviderResolver
   * @param astNotification
   * @return
   */
  // dynamic/static operation/source info
  @Override
  public Optional<MetadataQueryResult> perform(MuleApplicationNode muleApplicationNode,
                                               DataSenseProviderResolver dataSenseProviderResolver,
                                               AstNotification astNotification) {
    return muleApplicationNode
        .findMessageProcessorNode(location)
        .map(messageProcessorNode -> new MetadataQueryResult(
                                                             messageProcessorNode
                                                                 .getAnnotation(OperationModelAnnotation.class)
                                                                 .map(operationModelAnnotation -> enrichOperationModel(messageProcessorNode,
                                                                                                                       operationModelAnnotation
                                                                                                                           .getOperationModel(),
                                                                                                                       dataSenseProviderResolver,
                                                                                                                       astNotification))
                                                                 .orElse(null),
                                                             messageProcessorNode
                                                                 .getAnnotation(SourceModelAnnotation.class)
                                                                 .map(sourceModelAnnotation -> enrichSourceModel(messageProcessorNode,
                                                                                                                 sourceModelAnnotation
                                                                                                                     .getSourceModel(),
                                                                                                                 dataSenseProviderResolver,
                                                                                                                 astNotification))
                                                                 .orElse(null)));
  }
}
