/*
 * Copyright 2024 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metrics.impl.instrument;

import static java.util.Optional.ofNullable;

import org.mule.metrics.api.instrument.LongGauge;
import org.mule.metrics.api.instrument.builder.InstrumentBuilder;
import org.mule.metrics.api.instrument.builder.LongGaugeBuilder;
import org.mule.metrics.api.meter.Meter;
import org.mule.metrics.exporter.api.MeterExporter;
import org.mule.metrics.impl.instrument.repository.InstrumentRepository;

import java.util.function.Supplier;

/**
 * The {@code DefaultLongGauge} class is an implementation of the {@link LongGauge} interface. This class represents a gauge that
 * measures and retrieves long values, typically used in performance monitoring or metrics collection. It provides a builder
 * pattern for constructing instances with customizable attributes.
 */
public class DefaultLongGauge implements LongGauge {

  private final String name;
  private final String description;
  private final String unit;
  private final Meter meter;
  private final Supplier<Long> valueSupplier;

  /**
   * Private constructor for creating a {@code DefaultLongGauge} instance.
   *
   * @param name          the name of the gauge.
   * @param description   a brief description of the gauge.
   * @param unit          the unit of the gauge value.
   * @param meter         the {@link Meter} to which this gauge belongs.
   * @param valueSupplier a {@link Supplier} that provides the current value for the gauge.
   */
  private DefaultLongGauge(String name, String description, String unit, Meter meter,
                           Supplier<Long> valueSupplier) {
    this.name = name;
    this.description = description;
    this.unit = unit;
    this.meter = meter;
    this.valueSupplier = valueSupplier;
  }

  /**
   * Creates a new {@link DefaultLongGaugeBuilder} for building a {@code DefaultLongGauge} instance.
   *
   * @param gaugeName    the name of the gauge.
   * @param defaultMeter the {@link Meter} that this gauge will belong to.
   * @return a new {@link DefaultLongGaugeBuilder} instance.
   */
  public static DefaultLongGaugeBuilder builder(String gaugeName, Meter defaultMeter) {
    return new DefaultLongGaugeBuilder(gaugeName, defaultMeter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return name;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getDescription() {
    return description;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Meter getMeter() {
    return meter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getValueAsLong() {
    return valueSupplier.get();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getUnit() {
    return unit;
  }

  /**
   * The {@code DefaultLongGaugeBuilder} class implements the {@link LongGaugeBuilder} interface and provides a fluent builder
   * pattern for creating {@link DefaultLongGauge} instances.
   */
  public static class DefaultLongGaugeBuilder implements LongGaugeBuilder {

    private final String name;
    private final Meter meter;
    private String description;
    private String unit;
    private Supplier<Long> valueSupplier;
    private MeterExporter meterExporter;
    private InstrumentRepository instrumentRepository;

    /**
     * Creates a new {@code DefaultLongGaugeBuilder} with the specified name and meter.
     *
     * @param name  the name of the gauge.
     * @param meter the {@link Meter} associated with the gauge.
     */
    public DefaultLongGaugeBuilder(String name, Meter meter) {
      this.name = name;
      this.meter = meter;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public LongGaugeBuilder withValueSupplier(Supplier<Long> valueSupplier) {
      this.valueSupplier = valueSupplier;
      return this;
    }

    /**
     * Specifies the description for the gauge.
     *
     * @param description a brief description of the gauge.
     * @return the current {@code DefaultLongGaugeBuilder} instance for method chaining.
     */
    @Override
    public InstrumentBuilder<LongGauge> withDescription(String description) {
      this.description = description;
      return this;
    }

    /**
     * Specifies the unit of the gauge.
     *
     * @param unit a string representing the unit of measurement.
     * @return the current {@code DefaultLongGaugeBuilder} instance for method chaining.
     */
    @Override
    public InstrumentBuilder<LongGauge> withUnit(String unit) {
      this.unit = unit;
      return this;
    }

    /**
     * Specifies a {@link MeterExporter} to enable export of the built {@code LongGauge}.
     *
     * @param meterExporter the {@code MeterExporter} to enable export.
     * @return the current {@code DefaultLongGaugeBuilder} instance.
     */
    public DefaultLongGaugeBuilder withMeterExporter(MeterExporter meterExporter) {
      this.meterExporter = meterExporter;
      return this;
    }

    /**
     * Specifies an {@link InstrumentRepository} to register the built gauge.
     *
     * @param instrumentRepository the repository in which to register the gauge.
     * @return the current {@code DefaultLongGaugeBuilder} instance.
     */
    public DefaultLongGaugeBuilder withInstrumentRepository(InstrumentRepository instrumentRepository) {
      this.instrumentRepository = instrumentRepository;
      return this;
    }

    /**
     * Builds and returns a {@link LongGauge} instance.
     *
     * @return a new {@link LongGauge} instance.
     */
    @Override
    public LongGauge build() {
      LongGauge longGauge = ofNullable(instrumentRepository)
          .map(repository -> (LongGauge) repository.create(name, name -> doBuild()))
          .orElseGet(this::doBuild);

      if (meterExporter != null) {
        meterExporter.enableExport(longGauge);
      }

      return longGauge;
    }

    /**
     * Constructs the {@code DefaultLongGauge} instance.
     *
     * @return a new {@code DefaultLongGauge} instance.
     */
    private LongGauge doBuild() {
      return new DefaultLongGauge(name, description, unit, meter, valueSupplier);
    }
  }
}
