/**
* This module enables you to perform type introspection.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Types` to the header of your
* DataWeave script.
*/
@Since(version = "2.3.0")
%dw 2.0

/**
 * Represents a Qualified Name definition with a `localName` (a string) and a `namespace`.
 * If the QName does not have a Namespace, its value is `null`. 
 **/
@Since(version = "2.3.0")
type QName = {
    localName: String,
    namespace: Namespace | Null
}

/**
 * Represents an Attribute definition that is part of an Object field Key.
 */
@Since(version = "2.3.0")
type Attribute = {
    name: QName,
    required: Boolean,
    value: Type
}

/**
 * Represents a Field description that is part of an Object.
 */
@Since(version = "2.3.0")
type Field = {
    key: {
        name: QName,
        attributes: Array<Attribute>
    },
    required: Boolean,
    repeated: Boolean,
    value: Type
}

/**
 * Represents a Function parameter that is part of a Function type.
 */
@Since(version = "2.3.0")
type FunctionParam = {
   paramType: Type,
   optional: Boolean
}

/**
* Returns `true` if the input is the Boolean type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isBooleanType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ABoolean = Boolean
* output application/json
* ---
* {
*    a: isBooleanType(ABoolean),
*    b: isBooleanType(Boolean),
*    c: isBooleanType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isBooleanType(t:Type): Boolean = baseTypeOf(t) == Boolean

/**
* Returns `true` if the input is the Number type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isNumberType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ANumber = Number
* output application/json
* ---
* {
*    a: isNumberType(ANumber),
*    b: isNumberType(Number),
*    c: isNumberType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNumberType(t:Type): Boolean  = baseTypeOf(t) == Number

/**
* Returns `true` if the input is the Date type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isDateType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ADate = Date
* output application/json
* ---
* {
*    a: isDateType(ADate),
*    b: isDateType(Date),
*    c: isDateType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isDateType(t:Type): Boolean  = baseTypeOf(t) == Date

/**
* Returns `true` if the input is the DateTime type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isDateTimeType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ADateTime = DateTime
* output application/json
* ---
* {
*    a: isDateTimeType(ADateTime),
*    b: isDateTimeType(DateTime),
*    c: isDateTimeType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isDateTimeType(t:Type): Boolean  = baseTypeOf(t) == DateTime

/**
* Returns `true` if the input is the LocalDateTime type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isLocalDateTimeType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ALocalDateTime = LocalDateTime
* output application/json
* ---
* {
*    a: isLocalDateTimeType(ALocalDateTime),
*    b: isLocalDateTimeType(LocalDateTime),
*    c: isLocalDateTimeType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isLocalDateTimeType(t:Type): Boolean  = baseTypeOf(t) == LocalDateTime

/**
* Returns `true` if the input is the Time type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isTimeType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ATime = Time
* output application/json
* ---
* {
*    a: isTimeType(ATime),
*    b: isTimeType(Time),
*    c: isTimeType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isTimeType(t:Type): Boolean  = baseTypeOf(t) == Time

/**
* Returns `true` if the input is the LocalTime type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isLocalTimeType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ALocalTime = LocalTime
* output application/json
* ---
* {
*    a: isLocalTimeType(ALocalTime),
*    b: isLocalTimeType(LocalTime),
*    c: isLocalTimeType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isLocalTimeType(t:Type): Boolean  = baseTypeOf(t) == LocalTime

/**
* Returns `true` if the input is the TimeZone type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isTimeZoneType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ATimeZone = TimeZone
* output application/json
* ---
* {
*    a: isTimeZoneType(ATimeZone),
*    b: isTimeZoneType(TimeZone),
*    c: isTimeZoneType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isTimeZoneType(t:Type): Boolean  = baseTypeOf(t) == TimeZone

/**
* Returns `true` if the input is the Uri type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isUriType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AUri = Uri
* output application/json
* ---
* {
*    a: isUriType(AUri),
*    b: isUriType(Uri),
*    c: isUriType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isUriType(t:Type): Boolean  = baseTypeOf(t) == Uri

/**
* Returns `true` if the input is the Namespace type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isNamespaceType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ANamespace = Namespace
* output application/json
* ---
* {
*    a: isNamespaceType(ANamespace),
*    b: isNamespaceType(Namespace),
*    c: isNamespaceType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNamespaceType(t:Type): Boolean  = baseTypeOf(t) == Namespace


/**
* Returns `true` if the input is the Range type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isRangeType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ARange = Range
* output application/json
* ---
* {
*    a: isRangeType(ARange),
*    b: isRangeType(Range),
*    c: isRangeType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isRangeType(t:Type): Boolean  = baseTypeOf(t) == Range


/**
* Returns `true` if the input is the Period type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isPeriodType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type APeriod = Period
* output application/json
* ---
* {
*    a: isPeriodType(APeriod),
*    b: isPeriodType(Period),
*    c: isPeriodType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isPeriodType(t:Type) : Boolean = baseTypeOf(t) == Period


/**
* Returns `true` if the input is the Binary type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isBinaryType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ABinary = Binary
* output application/json
* ---
* {
*    a: isBinaryType(ABinary),
*    b: isBinaryType(Binary),
*    c: isBinaryType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isBinaryType(t:Type): Boolean  = baseTypeOf(t) == Binary


/**
* Returns `true` if the input is the Null type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isNullType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ANull = Null
* output application/json
* ---
* {
*    a: isNullType(ANull),
*    b: isNullType(Null),
*    c: isNullType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNullType(t:Type): Boolean  = baseTypeOf(t) == Null


/**
* Returns `true` if the input is the Nothing type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isNothingType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ANothing = Nothing
* output application/json
* ---
* {
*    a: isNothingType(ANothing),
*    b: isNothingType(Nothing),
*    c: isNothingType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNothingType(t:Type): Boolean  = baseTypeOf(t) == Nothing


/**
* Returns `true` if the input is the Any type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isAnyType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AAny = Any
* output application/json
* ---
* {
*    a: isAnyType(AAny),
*    b: isAnyType(Any),
*    c: isAnyType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isAnyType(t:Type): Boolean  = baseTypeOf(t) == Any


/**
* Returns `true` if the input is the Type type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isTypeType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = Type
* output application/json
* ---
* {
*    a: isTypeType(AType),
*    b: isTypeType(Type),
*    c: isTypeType(String),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isTypeType(t:Type) : Boolean = baseTypeOf(t) == Type


/**
* Returns `true` if the input is the String type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isStringType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AString = String
* output application/json
* ---
* {
*    a: isStringType(AString),
*    b: isStringType(String),
*    c: isStringType(Boolean),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isStringType(t:Type) : Boolean = baseTypeOf(t) == String


/**
* Returns `true` if the input is the Regex type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isRegexType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ARegex = Regex
* output application/json
* ---
* {
*    a: isRegexType(ARegex),
*    b: isRegexType(Regex),
*    c: isRegexType(Boolean),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isRegexType(t:Type): Boolean  = baseTypeOf(t) == Regex


/**
* Returns `true` if the input is the Key type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isKeyType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AKey = Key
* output application/json
* ---
* {
*    a: isKeyType(AKey),
*    b: isKeyType(Key),
*    c: isKeyType(Boolean),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isKeyType(t:Type): Boolean  = baseTypeOf(t) == Key


/**
* Returns `true` if the input is the Literal type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isLiteralType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ALiteralType = "Mariano"
* output application/json
* ---
* {
*    a: isLiteralType(ALiteralType),
*    b: isLiteralType(Boolean)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isLiteralType(t: Type):Boolean = native("system::IsLiteralTypeFunction")


/**
* Returns `true` if the input is the Function type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isFunctionType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AFunction = (String) -> String
* output application/json
* ---
* {
*    a: isFunctionType(AFunction),
*    b: isFunctionType(Boolean)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isFunctionType(t:Type): Boolean  = native("system::IsFunctionTypeFunction")


/**
* Returns the value of an input belongs to the Literal type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `literalValueOf` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = "Mariano"
* output application/json
* ---
* {
*    a: literalValueOf(AType)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "Mariano"
* }
* ----
**/
@Since(version = "2.3.0")
fun literalValueOf(t:Type): String | Number | Boolean = native("system::LiteralTypeValueFunction")



/**
* Returns `true` if the input is the Object type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isObjectType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = {name: String}
* output application/json
* ---
* {
*    a: isObjectType(AType),
*    b: isObjectType(Boolean),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isObjectType(t:Type): Boolean = native("system::IsObjectTypeFunction")


/**
* Returns `true` if the input type is the Array type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isArrayType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = Array<String>
* output application/json
* ---
* {
*    a: isArrayType(AType),
*    b: isArrayType(Boolean),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isArrayType(t:Type) : Boolean = native("system::IsArrayTypeFunction")


/**
* Returns `true` if the input type is the Union type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isUnionType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = String | Number
* output application/json
* ---
* {
*    a: isUnionType(AType),
*    b: isUnionType(Boolean),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isUnionType(t:Type): Boolean = native("system::IsUnionTypeFunction")



/**
* Returns `true` if the input type is the Intersection type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isIntersectionType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = {name: String} & {age: Number}
* output application/json
* ---
* {
*    a: isIntersectionType(AType),
*    b: isIntersectionType(Boolean),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isIntersectionType(t:Type):Boolean = native("system::IsIntersectionTypeFunction")

/**
* Returns an array of all the types that define a given Union type.
* This function fails if the input is not a Union type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `unionItems` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = String | Number
* output application/json
* ---
* {
*    a: unionItems(AType)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": ["String","Number"]
* }
* ----
**/
@Since(version = "2.3.0")
fun unionItems(t:Type): Array<Type>  = native("system::UnionTypeItemsFunction")


/**
*  Returns `true` if the input type is a Reference type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `isReferenceType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::core::Types
*  type AArray = Array<String> {n: 1}
*  type AArray2 = Array<AArray>
*  ---
*  {
*      a: isReferenceType( AArray),
*      b: isReferenceType(arrayItem(AArray2)),
*      c: isReferenceType(String)
*  }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": false,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isReferenceType(t: Type):Boolean = native("system::IsReferenceTypeFunction")

/**
* Returns the name of the input type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to query
* |===
*
* === Example
*
* This example shows how `nameOf` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::core::Types
* type AArray = Array<String> {n: 1}
* type AArray2 = Array<String>
* ---
*  {
*      a: nameOf(AArray),
*      b: nameOf(AArray2),
*      c: nameOf(String)
*  }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "a": "AArray",
*    "b": "AArray2",
*    "c": "String"
*  }
* ----
**/
@Since(version = "2.3.0")
fun nameOf(t: Type): String = native("system::NameOfFunction")

/**
* Returns an array of all the types that define a given Intersection type.
* This function fails if the input is not an Intersection type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `intersectionItems` behaves with different inputs.
* Note that the `AType` variable defines an Intersection type
* `{name: String} & {age: Number}` by using an `&amp;` between
* the two objects.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = {name: String} & {age: Number}
* output application/json
* ---
* {
*    a: intersectionItems(AType)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": ["Object","Object"]
* }
* ----
**/
@Since(version = "2.3.0")
fun intersectionItems(t:Type): Array<Type>  = native("system::IntersectionTypeItemsFunction")

/**
* Returns the type of the given array. This function fails if the input is not an Array type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `arrayItem` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type ArrayOfString = Array<String>
* type ArrayOfNumber = Array<Number>
* type ArrayOfAny = Array<Any>
* type ArrayOfAnyDefault = Array
* output application/json
* ---
* {
*    a: arrayItem(ArrayOfString),
*    b: arrayItem(ArrayOfNumber),
*    c: arrayItem(ArrayOfAny),
*    d: arrayItem(ArrayOfAnyDefault)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "String",
*   "b": "Number",
*   "c": "Any",
*   "d": "Any"
* }
* ----
**/
@Since(version = "2.3.0")
fun arrayItem(t: Type): Type = native("system::ArrayItemTypeFunction")

//TODO: DO WE WANT TO HAVE ANOTHER EXAMPLE?
/**
* Returns an the base type of the given type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `baseTypeOf` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = String {format: "YYYY-MM-dd"}
* output application/json
* ---
* {
*    a: baseTypeOf(AType)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "String"
* }
* ----
**/
@Since(version = "2.3.0")
fun baseTypeOf(t:Type): Type = native("system::BaseTypeFunction")


/**
* Returns metadata that is attached to the given type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to check.
* |===
*
* === Example
*
* This example shows how `metadataOf` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
* type AType = String {format: "YYYY-MM-dd"}
* output application/json
* ---
* {
*    a: metadataOf(AType)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": {"format": "YYYY-MM-dd"}
* }
* ----
**/
@Since(version = "2.3.0")
fun metadataOf(t:Type) : {} = native("system::MetadataOfFunction")

/**
* Returns the list of parameters from the given function type.
* This function fails if the provided type is not a Function type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The function type.
* |===
*
* === Example
*
* This example shows how `functionParamTypes` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::core::Types
* type AFunction = (String, Number) -> Number
* type AFunction2 = () -> Number
* ---
* {
*     a: functionParamTypes(AFunction),
*     b: functionParamTypes(AFunction2)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "a": [
*      {
*        "paramType": "String",
*        "optional": false
*      },
*      {
*        "paramType": "Number",
*        "optional": false
*      }
*    ],
*    "b": [
*
*    ]
*  }
* ----
**/
@Since(version = "2.3.0")
fun functionParamTypes(t: Type): Array<FunctionParam> = native("system::FunctionParamTypesFunction")


/**
* Returns the type of a function's return type.
* This function fails if the input type is not a Function type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The function type.
* |===
*
* === Example
*
* This example shows how `functionReturnType` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::core::Types
* type AFunction = (String, Number) -> Number
* type AFunction2 = () -> Number
* ---
* {
*     a: functionReturnType(AFunction),
*     b: functionReturnType(AFunction2)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "Number",
*   "b": "Number"
* }
* ----
**/
@Since(version = "2.3.0")
fun functionReturnType(t: Type): Type | Null = native("system::FunctionReturnTypeFunction")

/**
* Returns the array of fields from the given Object type.
* This function fails if the type is not an Object type.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The function type.
* |===
*
* === Example
*
* This example shows how `objectFields` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* import * from dw::core::Types
* ns ns0 http://acme.com
* type ADictionary = {_ : String}
* type ASchema = {ns0#name @(ns0#foo: String): {}}
* type AUser = {name @(foo?: String,l: Number)?: String, lastName*: Number}
* ---
* {
*     a: objectFields(ADictionary),
*     b: objectFields(ASchema),
*     c: objectFields(Object),
*     d: objectFields(AUser)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": [
*     {
*       "key": {
*         "name": {
*           "localName": "_",
*           "namespace": null
*         },
*         "attributes": [
*
*         ]
*       },
*       "required": true,
*       "repeated": false,
*       "value": "String"
*     }
*   ],
*   "b": [
*     {
*       "key": {
*         "name": {
*           "localName": "name",
*           "namespace": "http://acme.com"
*         },
*         "attributes": [
*           {
*             "name": {
*               "localName": "foo",
*               "namespace": "http://acme.com"
*             },
*             "value": "String",
*             "required": true
*           }
*         ]
*       },
*       "required": true,
*       "repeated": false,
*       "value": "Object"
*     }
*   ],
*   "c": [
*
*   ],
*   "d": [
*     {
*       "key": {
*         "name": {
*           "localName": "name",
*           "namespace": null
*         },
*         "attributes": [
*           {
*             "name": {
*               "localName": "foo",
*               "namespace": null
*             },
*             "value": "String",
*             "required": false
*           },
*           {
*             "name": {
*               "localName": "l",
*               "namespace": null
*             },
*             "value": "Number",
*             "required": true
*           }
*         ]
*       },
*       "required": false,
*       "repeated": false,
*       "value": "String"
*     },
*     {
*       "key": {
*         "name": {
*           "localName": "lastName",
*           "namespace": null
*         },
*         "attributes": [
*
*         ]
*       },
*       "required": true,
*       "repeated": true,
*       "value": "Number"
*     }
*   ]
* }
* ----
**/
@Since(version = "2.3.0")
fun objectFields(t: Type): Array<Field>  = native("system::ObjectTypeFieldsFunction")
