package org.mule.weave.v2.module.excel

import org.apache.poi.ss.util.CellReference
import org.mule.weave.v2.module.option.BooleanModuleOption
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.module.option.OptionalDoubleModuleOption
import org.mule.weave.v2.module.option.OptionalIntModuleOption
import org.mule.weave.v2.module.option.OptionalStringModuleOption
import org.mule.weave.v2.module.option.Settings
import org.mule.weave.v2.module.option.StringModuleOption
import org.mule.weave.v2.module.reader.ConfigurableDeferred
import org.mule.weave.v2.module.reader.ConfigurableStreaming
import org.mule.weave.v2.parser.location.UnknownLocation

import scala.util.matching.Regex

class ExcelSettings extends Settings {
  var header: Boolean = _
  var ignoreEmptyLine: Boolean = _
  var tableOffset: Option[String] = _
  var _headerColNum: Int = 0
  var _headerRowNum: Int = 0
  var zipBombCheck: Boolean = true
  var minInflateRatio: Option[Double] = _
  var maxEntrySize: Option[Int] = _

  /**
    * By default it's 0
    */
  def headerColNum() = {
    if (_headerColNum == -1)
      _headerColNum = 0
    _headerColNum
  }

  /**
    * By default it's 0
    */
  def headerRowNum() = {
    if (_headerRowNum == -1)
      _headerRowNum = 0
    _headerRowNum
  }

  def bodyRowNum: Int = if (header) headerRowNum + 1 else headerRowNum()

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions ++
      Map(
        BooleanModuleOption("header", defaultValue = true, descriptionUrl = "data-format/excel/header.asciidoc"),
        BooleanModuleOption("ignoreEmptyLine", defaultValue = true, descriptionUrl = "data-format/excel/ignoreEmptyLine.asciidoc"),
        OptionalStringModuleOption("tableOffset", None, descriptionUrl = "data-format/excel/tableOffset.asciidoc"),
        BooleanModuleOption("zipBombCheck", defaultValue = true, descriptionUrl = "data-format/excel/zipBombCheck.asciidoc").markAsSecurity(),
        OptionalDoubleModuleOption("minInflateRatio", descriptionUrl = "data-format/excel/minInflateRatio.asciidoc").markAsSecurity(),
        OptionalIntModuleOption("maxEntrySize", maxValue = Some(Int.MaxValue), descriptionUrl = "data-format/excel/maxEntrySize.asciidoc").markAsSecurity())
  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case "header" => {
        this.header = value.asInstanceOf[Boolean]
      }
      case "ignoreEmptyLine" => {
        this.ignoreEmptyLine = value.asInstanceOf[Boolean]
      }
      case "zipBombCheck" => {
        this.zipBombCheck = value.asInstanceOf[Boolean]
      }
      case "tableOffset" => {
        this.tableOffset = value.asInstanceOf[Option[String]]
        setHeaderPosition()
      }
      case "minInflateRatio" =>
        this.minInflateRatio = value.asInstanceOf[Option[Double]]
      case "maxEntrySize" =>
        this.maxEntrySize = value.asInstanceOf[Option[Int]]
      case _ => super.writeSettingsValue(settingName, value)
    }
  }

  private def shouldCalculateHeader(): Boolean = {
    header && tableOffset.isDefined
  }

  private def setHeaderPosition(): Unit = {
    if (shouldCalculateHeader()) {
      val pattern: Regex = "([A-Z]+)([0-9]+)".r
      val offset: String = tableOffset.get
      val matches: Boolean = pattern.pattern.matcher(offset).matches()
      if (matches) {
        val pattern(colStr, rowStr) = offset
        _headerColNum = CellReference.convertColStringToIndex(colStr)
        _headerRowNum = rowStr.toInt - 1
      } else {
        throw new ExcelReadingException(UnknownLocation, offset)
      }
    }
  }

}

class ExcelWriterSettings extends ExcelSettings with ConfigurableDeferred {}

class ExcelReaderSettings extends ExcelSettings with ConfigurableStreaming {

  var tableLimit: String = TableLimit.UNBOUNDED

  override def loadSettingsOptions(): Map[String, ModuleOption] =
    super.loadSettingsOptions ++
      Map(
        StringModuleOption(
          "tableLimit",
          TableLimit.UNBOUNDED,
          descriptionUrl = "data-format/excel/tableLimit.asciidoc"))

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case "tableLimit" => {
        this.tableLimit = value.asInstanceOf[String]
      }
      case _ => super.writeSettingsValue(settingName, value)
    }
  }
}

object TableLimit {
  val UNBOUNDED = "Unbounded"
  val HEADER_SIZE = "HeaderSize"

  def isUnbounded(text: String): Boolean = {
    text == UNBOUNDED
  }

  def isHeaderSize(text: String): Boolean = {
    text == HEADER_SIZE
  }
}
