package org.mule.weave.v2.module.pojo.reader

import org.mule.weave.v2.model.EvaluationContext

import java.lang.reflect.Method
import java.lang.reflect.Modifier
import java.util.concurrent.ConcurrentHashMap
import scala.annotation.tailrec
import scala.collection.JavaConverters._

class BeanDefinition(clazz: Class[_]) {
  private val requestedCache = new ConcurrentHashMap[String, PropertyDefinition](16, 0.9f, 1)
  private val readAccessorsCache = new ConcurrentHashMap[String, PropertyDefinition](16, 0.9f, 1)
  private var fullyLoaded: Boolean = false

  def getProperty(name: String)(implicit ctx: EvaluationContext): PropertyDefinition = {
    //This method potentially puts invalid properties into the cache, so we created a "requestedCache" to separate them
    //Invalid properties are possible because manually queried properties (with a selector) go through here
    //That is to allow fields without accessors
    var maybeDefinition = readAccessorsCache.get(name)
    if (maybeDefinition == null) {
      maybeDefinition = requestedCache.get(name)
    }
    if (maybeDefinition == null) {
      val definition = PropertyDefinition(name, clazz)
      requestedCache.put(name, definition)
      definition
    } else {
      maybeDefinition
    }
  }

  def getAllProperties(implicit ctx: EvaluationContext): Iterable[PropertyDefinition] = {
    if (fullyLoaded) {
      readAccessorsCache.values.asScala
    } else {
      loadReadAccessors(clazz)
      fullyLoaded = true
      readAccessorsCache.values.asScala
    }
  }

  @tailrec
  private final def loadReadAccessors(clazz: Class[_])(implicit ctx: EvaluationContext): Unit = {
    val methods = clazz.getDeclaredMethods
    methods.foreach(method => {
      // skip static methods.
      if (isValidModifier(method)) {
        val parameterCount = method.getParameterTypes.length
        val name = method.getName
        parameterCount match {
          case 0 if name.startsWith(BeanConstants.GET_PREFIX) && name.substring(BeanConstants.GET_PREFIX.length) != "" =>
            cacheReadAccessors(NameGenerator.lowerJavaConvention(name.substring(BeanConstants.GET_PREFIX.length)))
          case 0 if name.startsWith(BeanConstants.IS_PREFIX) =>
            cacheReadAccessors(NameGenerator.lowerJavaConvention(name.substring(BeanConstants.IS_PREFIX.length)))
          case _ =>
        }
      }
    })
    val superclass = clazz.getSuperclass
    if (superclass != null && !classOf[Object].equals(superclass)) {
      loadReadAccessors(superclass)
    }
  }

  private def isValidModifier(method: Method): Boolean = {
    val mods = method.getModifiers
    Modifier.isPublic(mods) && !Modifier.isNative(mods) && !Modifier.isStatic(mods)
  }

  private def cacheReadAccessors(name: String)(implicit ctx: EvaluationContext): Unit = {
    val maybeDefinition = readAccessorsCache.get(name)
    if (maybeDefinition == null) {
      val definition = PropertyDefinition(name, clazz)
      readAccessorsCache.put(name, definition)
    }
  }
}
