package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.api.tooling.annotation.DWAnnotationProcessor
import org.mule.weave.v2.api.tooling.annotation.DWCanonicalProcessingPhase
import org.mule.weave.v2.api.tooling.annotation.DWParsingProcessingPhase
import org.mule.weave.v2.api.tooling.annotation.DWScopeProcessingPhase
import org.mule.weave.v2.api.tooling.annotation.DWTypeProcessingPhase
import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.NameIdentifier.CORE_MODULE
import org.mule.weave.v2.parser.module.ModuleLoaderProvider
import org.mule.weave.v2.parser.phase.listener.ParsingNotificationManager
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.versioncheck.SVersion

import scala.collection.mutable

/**
  * This class contains all necesary information for running all the parsing phases.
  * This object will be passed to each phase.
  * For Example contains the message collector, then name of the element that is being parsed or information such as implicit inputs or implicit imports.
  */
@WeaveApi(Seq("data-weave-agent"))
class ParsingContext(
  val nameIdentifier: NameIdentifier,
  val messageCollector: MessageCollector,
  val moduleParserManager: ModuleParsingPhasesManager,
  val parent: Option[ParsingContext] = None,
  val strictMode: Boolean = true,
  val errorTrace: Int = Grammar.DEFAULT_ERROR_TRACE,
  val attachDocumentation: Boolean = true,
  val notificationManager: ParsingNotificationManager = new ParsingNotificationManager(),
  var languageLevel: Option[SVersion] = None,
  val settings: BaseParsingContextSettings = DefaultParsingContextSettings,
  val annotationProcessorProvider: AnnotationProcessorRegistry = DefaultAnnotationProcessorRegistry) {

  //  var implicitImports: Seq[String] = Seq()
  var implicitImports: Seq[String] = Seq(ParsingContext.CORE_MODULE)

  var skipVerification: Boolean = false

  //Mapping implicits
  var implicitInput: Seq[ImplicitInputData] = Seq()
  var expectedOutputType: Option[WeaveType] = None
  var implicitOutputMimeType: Option[String] = None

  val typeHelper: TypeHelper = new TypeHelper(notificationManager)

  private val annotationProcessors: mutable.Map[NameIdentifier, DWAnnotationProcessor] = mutable.Map(annotationProcessorProvider.annotations().toSeq: _*)

  private var commonSubExpressionElimination: Boolean = true

  private var constantFoldingPhase: Boolean = true

  def disableCommonSubExpressionElimination(): Unit = {
    commonSubExpressionElimination = false
  }

  def shouldRunCommonSubExpressionElimination(): Boolean = {
    commonSubExpressionElimination
  }

  def disableConstantFoldingPhase(): Unit = {
    constantFoldingPhase = false
  }

  def shouldRunConstantFoldingPhase(): Boolean = {
    constantFoldingPhase
  }

  def registerAnnotationProcessor(nameIdentifier: NameIdentifier, processor: DWAnnotationProcessor): ParsingContext = {
    if (processor.getProcessingPhase == DWParsingProcessingPhase) {
      annotationProcessors.put(nameIdentifier.localName(), processor)
    } else {
      annotationProcessors.put(nameIdentifier, processor)
    }
    this
  }

  def hasParsingPhaseAnnotationProcessors: Boolean = {
    annotationProcessors.values.exists(_.getProcessingPhase == DWParsingProcessingPhase)
  }

  def hasCanonicalPhaseAnnotationProcessors: Boolean = {
    annotationProcessors.values.exists(_.getProcessingPhase == DWCanonicalProcessingPhase)
  }

  def hasTypePhaseAnnotationProcessors: Boolean = {
    annotationProcessors.values.exists(_.getProcessingPhase == DWTypeProcessingPhase)
  }

  def addImplicitInput(name: String, weaveType: Option[WeaveType]): ParsingContext = {
    addImplicitInput(ImplicitInputData(name, weaveType, None))
  }

  def withLanguageLevel(version: SVersion): ParsingContext = {
    languageLevel = Some(version)
    this
  }

  def addImplicitInput(input: ImplicitInputData): ParsingContext = {
    implicitInput = implicitInput :+ input
    this
  }

  def addImplicitImport(module: String): ParsingContext = {
    implicitImports = implicitImports :+ module
    this
  }

  def typePhaseAnnotationProcessorFor(annotationName: NameIdentifier): Option[DWAnnotationProcessor] = {
    annotationProcessors.get(annotationName).filter(_.getProcessingPhase == DWTypeProcessingPhase)
  }

  def scopePhaseAnnotationProcessorFor(annotationName: NameIdentifier): Option[DWAnnotationProcessor] = {
    annotationProcessors.get(annotationName).filter(_.getProcessingPhase == DWScopeProcessingPhase)
  }

  def canonicalPhaseAnnotationProcessorFor(annotationName: NameIdentifier): Option[DWAnnotationProcessor] = {
    annotationProcessors.get(annotationName).filter(_.getProcessingPhase == DWCanonicalProcessingPhase)
  }

  def parsingPhaseAnnotationProcessorFor(annotationName: NameIdentifier): Option[DWAnnotationProcessor] = {
    annotationProcessors.get(annotationName.localName()).filter(_.getProcessingPhase == DWParsingProcessingPhase)
  }

  def detectCircularImport(name: NameIdentifier): Boolean = {
    parent match {
      case None              => false
      case Some(parentValue) => parentValue.nameIdentifier == name || parentValue.detectCircularImport(name)
    }
  }

  def getCircularImport(name: NameIdentifier): Seq[NameIdentifier] = {
    val rest = parent match {
      case Some(value) =>
        if (value.nameIdentifier == name) {
          Seq(value.nameIdentifier)
        } else {
          value.getCircularImport(name)
        }
      case None => Seq()
    }
    nameIdentifier +: rest
  }

  def getModuleNodeForModule(name: NameIdentifier): PhaseResult[ParsingResult[ModuleNode]] = {
    val result = tryToParseModule(name)
    result.foreach((result) => messageCollector.mergeWith(result.messages()))
    result
      .getOrElse({
        FailureResult(this)
      })
  }

  def tryToParseModule(name: NameIdentifier): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    moduleParserManager.parseModule(name, this)
  }

  def getScopeGraphForModule(name: NameIdentifier): PhaseResult[ScopeGraphResult[ModuleNode]] = {
    val result = moduleParserManager.scopeCheckModule(name, this)
    result.foreach((result) => messageCollector.mergeWith(result.messages()))
    result
      .getOrElse({
        FailureResult(this)
      })
  }

  def getTypeCheckingForModule(name: NameIdentifier): PhaseResult[TypeCheckingResult[ModuleNode]] = {
    val result = moduleParserManager.typeCheckModule(name, this)
    result.foreach((result) => messageCollector.mergeWith(result.messages()))
    result
      .getOrElse({
        FailureResult(this)
      })
  }

  def child(resource: NameIdentifier): ParsingContext = {
    val parsingContext = new ParsingContext(resource, MessageCollector(), moduleParserManager, Some(this), strictMode, errorTrace, attachDocumentation, notificationManager, languageLevel, settings)
    copyElements(parsingContext)
    parsingContext
  }

  def withMessageCollector(messageCollector: MessageCollector): ParsingContext = {
    val parsingContext = new ParsingContext(nameIdentifier, messageCollector, moduleParserManager, parent, strictMode, errorTrace, attachDocumentation, notificationManager, languageLevel, settings)
    copyElements(parsingContext)
    parsingContext
  }

  def withNameIdentifier(nameIdentifier: NameIdentifier): ParsingContext = {
    val parsingContext = new ParsingContext(nameIdentifier, MessageCollector(), moduleParserManager, parent, strictMode, errorTrace, attachDocumentation, notificationManager, languageLevel, settings)
    copyElements(parsingContext)
    parsingContext
  }

  private def copyElements(parsingContext: ParsingContext): Unit = {
    parsingContext.implicitImports = implicitImports
    parsingContext.implicitInput = implicitInput
    parsingContext.skipVerification = skipVerification
    parsingContext.expectedOutputType = expectedOutputType
    parsingContext.implicitOutputMimeType = implicitOutputMimeType
    parsingContext.commonSubExpressionElimination = commonSubExpressionElimination
    parsingContext.annotationProcessors.++=(annotationProcessors)
  }
}

object ParsingContext {

  val CORE_MODULE: String = "dw::Core"

  def apply(nameIdentifier: NameIdentifier, resourceResolver: WeaveResourceResolver, moduleLoaderProvider: ModuleLoaderProvider, messageCollector: MessageCollector = new MessageCollector()): ParsingContext = {
    new ParsingContext(nameIdentifier, messageCollector, ModuleParsingPhasesManager(ModuleLoaderManager(Seq(ModuleLoader(resourceResolver)), moduleLoaderProvider)))
  }

  def apply(nameIdentifier: NameIdentifier, messageCollector: MessageCollector, parser: ModuleParsingPhasesManager): ParsingContext = {
    new ParsingContext(nameIdentifier, messageCollector, parser)
  }

  def apply(nameIdentifier: NameIdentifier, messageCollector: MessageCollector, parser: ModuleParsingPhasesManager, errorTrace: Int): ParsingContext = {
    new ParsingContext(nameIdentifier, messageCollector, parser, errorTrace = errorTrace)
  }

  def apply(nameIdentifier: NameIdentifier, messageCollector: MessageCollector, parser: ModuleParsingPhasesManager, strictMode: Boolean): ParsingContext = {
    new ParsingContext(nameIdentifier, messageCollector, parser, strictMode = strictMode)
  }

  def apply(nameIdentifier: NameIdentifier, parser: ModuleParsingPhasesManager): ParsingContext = {
    new ParsingContext(nameIdentifier, new MessageCollector, parser)
  }

  def apply(nameIdentifier: NameIdentifier, parser: ModuleParsingPhasesManager, annotationProcessorProvider: AnnotationProcessorRegistry): ParsingContext = {
    new ParsingContext(nameIdentifier, new MessageCollector, parser, annotationProcessorProvider = annotationProcessorProvider)
  }

  def apply(nameIdentifier: NameIdentifier, parser: ModuleParsingPhasesManager, errorTrace: Int, attachDocumentation: Boolean): ParsingContext = {
    new ParsingContext(nameIdentifier, new MessageCollector, parser, errorTrace = errorTrace, attachDocumentation = attachDocumentation)
  }

  def apply(nameIdentifier: NameIdentifier, parser: ModuleParsingPhasesManager, errorTrace: Int, attachDocumentation: Boolean, settings: BaseParsingContextSettings): ParsingContext = {
    new ParsingContext(nameIdentifier, new MessageCollector, parser, errorTrace = errorTrace, attachDocumentation = attachDocumentation, settings = settings)
  }

  def apply(nameIdentifier: NameIdentifier, parser: ModuleParsingPhasesManager, errorTrace: Int, attachDocumentation: Boolean, notificationManager: ParsingNotificationManager): ParsingContext = {
    new ParsingContext(nameIdentifier, new MessageCollector, parser, errorTrace = errorTrace, attachDocumentation = attachDocumentation, notificationManager = notificationManager)
  }

  def apply(nameIdentifier: NameIdentifier, parser: ModuleParsingPhasesManager, strictMode: Boolean): ParsingContext = {
    new ParsingContext(nameIdentifier, new MessageCollector, parser, strictMode = strictMode)
  }

}
