package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.ast.QName

import scala.util.Failure
import scala.util.Success
import scala.util.Try

/**
  * Handles the coercion of Weave Types
  */
object TypeCoercer {

  /**
    * Tries to coerce the actual type to the expected one
    *
    * @param expected The expected
    * @param actual   The actual type
    * @param ctx      The context
    * @return Returns Some if the coercion exists if not None
    */
  def coerce(expected: WeaveType, actual: WeaveType, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    // If they are structural matching no need to coerce
    val maybeCoerced = if (TypeHelper.canBeAssignedTo(actual, expected, ctx)) {
      Some(actual)
    } else {
      actual match {
        case UnionType(of) =>
          val maybeTypes: Seq[WeaveType] = of.flatMap((actual: WeaveType) => coerce(expected, actual, ctx))
          if (maybeTypes.isEmpty) {
            None
          } else {
            Some(TypeHelper.unify(maybeTypes))
          }
        case _: AnyType        => Some(expected)
        case rt: ReferenceType => coerce(expected, rt.resolveType(), ctx)
        case _ =>
          expected match {
            case UnionType(of) => {
              val validOptions = of.flatMap((expected: WeaveType) => coerce(expected, actual, ctx))
              if (validOptions.isEmpty) {
                None
              } else {
                Some(TypeHelper.unify(validOptions))
              }
            }
            case ArrayType(of) => {
              actual match {
                case _: RangeType => {
                  val itemType: Option[WeaveType] = coerce(of, NumberType(), ctx)
                  itemType.map(ArrayType)
                }
                case _ => None
              }
            }
            case TypeParameter(_, topType, baseType, _, _) => {
              topType.flatMap(coerce(_, actual, ctx)).orElse(baseType.flatMap(coerce(_, actual, ctx))).orElse(Some(actual))
            }
            case ObjectType(_, _, _) => {
              actual match {
                case _: NamespaceType => Some(namespaceAsObjectType)
                case obj: ObjectType if (obj.properties.isEmpty & !obj.close) => Some(expected)
                case _ => None
              }
            }
            case StringType(opVal) => {
              actual match {
                case bl: BooleanType => {
                  if (opVal.isDefined) {
                    if (bl.value.isDefined && opVal.get == bl.value.get.toString) {
                      Some(StringType(opVal))
                    } else if (bl.value.isEmpty) {
                      Some(StringType(opVal))
                    } else {
                      None
                    }
                  } else {
                    Some(StringType(bl.value.map(_.toString)))
                  }
                }
                case _: DateTimeType      => Some(StringType(opVal))
                case _: LocalDateTimeType => Some(StringType(opVal))
                case _: LocalTimeType     => Some(StringType(opVal))
                case _: LocalDateType     => Some(StringType(opVal))
                case _: TimeType          => Some(StringType(opVal))
                case _: PeriodType        => Some(StringType(opVal))
                case _: TimeZoneType      => Some(StringType(opVal))
                case nt: NumberType => {
                  if (opVal.isDefined) {
                    if (nt.value.isDefined && opVal.get == nt.value.get) {
                      Some(StringType(opVal))
                    } else {
                      None
                    }
                  } else {
                    Some(StringType(nt.value))
                  }
                }
                case nt: NamespaceType => {
                  if (opVal.isDefined) {
                    if (nt.namespace.isDefined && nt.namespace.get.value.isDefined && opVal.get == nt.namespace.get.value.get) {
                      Some(StringType(opVal))
                    } else {
                      None
                    }
                  } else {
                    Some(StringType(nt.namespace.flatMap(_.value)))
                  }
                }
                case _: UriType    => Some(StringType(opVal))
                case _: BinaryType => Some(StringType(opVal))
                case _: TypeType   => Some(StringType(opVal))
                case nt: NameType => {
                  if (nt.value.isDefined && !nt.value.get.matchesAllNs) {
                    if (opVal.isEmpty) {
                      Some(StringType(nt.value.map(_.name)))
                    } else if (opVal.isDefined) {
                      if (opVal.get == nt.value.get.name) {
                        Some(StringType(opVal))
                      } else {
                        None
                      }
                    } else {
                      Some(StringType(opVal))
                    }
                  } else {
                    Some(StringType(opVal))
                  }
                }
                case kt: KeyType => {
                  coerce(expected, kt.name, ctx)
                }
                case StringType(None) => Some(StringType(opVal))
                case _: RegexType     => Some(StringType(opVal))
                case _                => None
              }
            }
            case AnyType() => Some(AnyType())
            case BooleanType(opVal, _) => {
              actual match {
                case BooleanType(None, c) => Some(BooleanType(opVal, c))
                case st: StringType => {
                  if (st.value.isDefined) {
                    if (st.value.get.matches("(?i)^true$"))
                      if (opVal.isEmpty || (opVal.isDefined && opVal.get)) Some(BooleanType(Some(true))) else None
                    else if (st.value.get.matches("(?i)^false$"))
                      if (opVal.isEmpty || (opVal.isDefined && !opVal.get)) Some(BooleanType(Some(false))) else None
                    else
                      None
                  } else Some(BooleanType())
                }
                case _: NameType => Some(BooleanType(if (opVal.isDefined) opVal else None))
                case _: KeyType  => Some(BooleanType(if (opVal.isDefined) opVal else None))
                case _           => None
              }
            }
            case NumberType(opVal) => {
              actual match {
                case _: DateTimeType => Some(NumberType(opVal))
                case st: StringType => {
                  st.value match {
                    case Some(stValue) => {
                      Try(BigDecimal.apply(stValue)) match {
                        case Failure(_) => None
                        case Success(actualValue) => {
                          if (opVal.isDefined) {
                            Try(BigDecimal.apply(opVal.get)) match {
                              case Failure(_) => Some(NumberType(opVal)) //This shouldn't happen as a literal number should always be a number
                              case Success(expectedValue) => {
                                if (actualValue == expectedValue) {
                                  Some(NumberType(opVal))
                                } else {
                                  None
                                }
                              }
                            }
                          } else {
                            Some(NumberType(Some(stValue)))
                          }
                        }
                      }
                    }
                    case None => {
                      Some(NumberType(opVal))
                    }
                  }
                }
                case NumberType(None) => Some(NumberType(opVal))
                case _: NameType      => Some(NumberType(opVal))
                case _: KeyType       => Some(NumberType(opVal))
                case _: PeriodType    => Some(NumberType(opVal))
                case _                => None
              }
            }
            case NameType(_) => {
              actual match {
                case StringType(strVal) => Some(NameType(strVal.map(QName(_))))
                case NumberType(strVal) => Some(NameType(strVal.map(QName(_))))
                case KeyType(name, _)   => coerce(expected, name, ctx)
                case _                  => None
              }
            }
            case RegexType() =>
              actual match {
                case _: StringType => Some(RegexType())
                case _: NameType   => Some(RegexType())
                case _: KeyType    => Some(RegexType())
                case _             => None
              }
            case KeyValuePairType(_, _, _, _) => {
              actual match {
                case nv: NameValuePairType => Some(KeyValuePairType(KeyType(nv.name), nv.value))
                case _                     => None
              }
            }
            case KeyType(_, _) => {
              actual match {
                case nt: NameType    => Some(KeyType(nt))
                case st: StringType  => Some(KeyType(NameType(st.value.map(QName(_)))))
                case num: NumberType => Some(KeyType(NameType(num.value.map(QName(_)))))
                case _               => None
              }
            }
            case NameValuePairType(_, _, _) => {
              actual match {
                case kvp: KeyValuePairType if kvp.key.isInstanceOf[KeyType] => Some(NameValuePairType(kvp.key.asInstanceOf[KeyType].name, kvp.value))
                case _ => None
              }
            }
            case RangeType() => None
            case UriType(_) => {
              actual match {
                case StringType(strValue) => Some(UriType(strValue))
                case _                    => None
              }
            }
            case DateTimeType() => {
              actual match {
                case _: NumberType        => Some(DateTimeType())
                case _: DateTimeType      => Some(DateTimeType())
                case _: LocalDateTimeType => Some(DateTimeType())
                case _: StringType        => Some(DateTimeType())
                case _                    => None
              }
            }
            case LocalDateTimeType() => {
              actual match {
                case _: DateTimeType      => Some(LocalDateTimeType())
                case _: LocalDateTimeType => Some(LocalDateTimeType())
                case _: StringType        => Some(LocalDateTimeType())
                case _                    => None
              }
            }
            case LocalDateType() => {
              actual match {
                case _: LocalDateType     => Some(LocalDateType())
                case _: DateTimeType      => Some(LocalDateType())
                case _: LocalDateTimeType => Some(LocalDateType())
                case _: StringType        => Some(LocalDateType())
                case _                    => None
              }
            }
            case LocalTimeType() => {
              actual match {
                case _: LocalTimeType     => Some(LocalTimeType())
                case _: DateTimeType      => Some(LocalTimeType())
                case _: LocalDateTimeType => Some(LocalTimeType())
                case _: TimeType          => Some(LocalTimeType())
                case _: StringType        => Some(LocalTimeType())
                case _                    => None
              }
            }
            case TimeType() => {
              actual match {
                case _: LocalTimeType     => Some(TimeType())
                case _: DateTimeType      => Some(TimeType())
                case _: LocalDateTimeType => Some(TimeType())
                case _: StringType        => Some(TimeType())
                case _                    => None
              }
            }
            case TimeZoneType() => {
              actual match {
                case _: DateTimeType => Some(TimeZoneType())
                case _: TimeType     => Some(TimeZoneType())
                case _: StringType   => Some(TimeZoneType())
                case _               => None
              }
            }
            case PeriodType() => {
              actual match {
                case _: StringType => Some(PeriodType())
                case _             => None
              }
            }
            case BinaryType() => {
              actual match {
                case _: NumberType => Some(BinaryType())
                case _: StringType => Some(BinaryType())
                case _             => None
              }
            }
            case TypeType(_) => {
              actual match {
                case _: StringType => Some(TypeType(AnyType()))
                case _             => None
              }
            }
            case NamespaceType(_, _) => {
              actual match {
                case ob: ObjectType if isNamespaceObjectType(ob) => {
                  val prefix: StringType = ObjectTypeHelper.selectPropertyValue(QName("prefix"), ob).get.asInstanceOf[StringType]
                  val uri: StringType = ObjectTypeHelper.selectPropertyValue(QName("uri"), ob).get.asInstanceOf[StringType]
                  Some(NamespaceType(prefix.value, uri.value.map((value) => UriType(Some(value)))))
                }
                case _ => None
              }
            }
            case rt: ReferenceType => coerce(rt.resolveType(), actual, ctx)
            case _                 => None

          }
      }
    }
    maybeCoerced
  }

  private def isNamespaceObjectType(objectType: ObjectType): Boolean = {
    val prefix = ObjectTypeHelper.selectPropertyValue(QName("prefix"), objectType)
    val uri = ObjectTypeHelper.selectPropertyValue(QName("uri"), objectType)
    uri.isDefined && prefix.isDefined && uri.get.isInstanceOf[StringType] && uri.get.isInstanceOf[StringType]
  }

  private def namespaceAsObjectType: ObjectType = {
    ObjectType(Seq(KeyValuePairType(KeyType(NameType(Some(QName("prefix")))), StringType()), KeyValuePairType(KeyType(NameType(Some(QName("uri")))), StringType())))
  }
}
