package org.mule.weave.v2.inspector

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.inspector.ReferenceUtils.isReferencingTo
import org.mule.weave.v2.parser.ReduceObjectToDynamicObject
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.NameIdentifier.CORE_MODULE
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.TypeCheckingResult
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.WeaveType

object ReduceConcatObjectsInspector extends CodeInspector[TypeCheckingResult[_ <: AstNode]] {

  def isReferencingReduce(variable: NameIdentifier, scopeData: ScopesNavigator): Boolean = {
    isReferencingTo(variable, CORE_MODULE.::("reduce"), scopeData)
  }

  def isReferencingPlusPlus(variable: NameIdentifier, scopeData: ScopesNavigator): Boolean = {
    isReferencingTo(variable, CORE_MODULE.::("++"), scopeData)
  }

  override def inspect(node: AstNode, scopeData: TypeCheckingResult[_ <: AstNode], parsingContext: ParsingContext): Unit = {
    node match {
      case functionCallNode @ FunctionCallNode(VariableReferenceNode(name, _), args, _, _) if (isReferencingReduce(name, scopeData.scope) && args.args.size == 2) => {
        //Reduce function call
        val reduceLHS = args.args(0)
        val reduceRHS = args.args(1)
        scopeData.typeGraph.findNode(reduceLHS) match {
          case Some(node) => {
            val maybeType: Option[WeaveType] = node.resultType()
            maybeType match {
              case Some(typeOfNode) if (TypeHelper.isArrayType(typeOfNode, classOf[ObjectType])) => {
                reduceRHS match {
                  case fn: FunctionNode if (fn.params.paramList.size == 2 && fn.params.paramList.forall(_.defaultValue.isEmpty)) => {
                    fn.body match {
                      case FunctionCallNode(VariableReferenceNode(funName, _), args, _, _) if (args.args.size == 2 && isReferencingPlusPlus(funName, scopeData.scope)) => {
                        val leftH = args.args(0)
                        val rightH = args.args(1)
                        if ((referenceToParam(leftH, scopeData, fn, 1) && referenceToParam(rightH, scopeData, fn, 0))) {
                          parsingContext.messageCollector.warning(ReduceObjectToDynamicObject(reduceLHS, functionCallNode), functionCallNode.location())
                        }
                      }
                      case _ =>
                    }
                  }
                  case _ =>
                }
              }
              case _ =>
            }
          }
          case _ =>
        }
      }
      case _ =>
    }
  }

  private def referenceToParam(sourceNode: AstNode, scopeData: TypeCheckingResult[_ <: AstNode], fn: FunctionNode, paramIndex: Int) = {
    sourceNode match {
      case VariableReferenceNode(nameIdentifier: NameIdentifier, _) => {
        scopeData.scope.resolveVariable(nameIdentifier) match {
          case Some(ref) => {
            val targetParameter = fn.params.paramList(paramIndex).variable
            ref.referencedNode eq targetParameter
          }
          case _ => false
        }
      }
      case _ => false
    }
  }
}

class ReduceObjectToDynamicObjectAction(expression: AstNode, containerNode: AstNode) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {
    val startIndex = containerNode.location().startPosition.index
    document.delete(startIndex, containerNode.location().endPosition.index)
    document.insert("{\n\t(" + CodeGenerator.generate(expression) + ")\n}", startIndex)
  }
}
