package org.mule.weave.v2.ts

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.scope.AstNavigator

class TypeInformationProviderService(editor: WeaveEditorSupport) {

  /**
    * Returns the type of the given AstNode
    * @param astNode The astNode to search the type
    * @return The weave type if any
    */
  def typeOf(astNode: AstNode): Option[WeaveType] = {
    val value = editor.typeGraph()
    var result: Option[WeaveType] = None
    if (value.isDefined) {
      val navigator = editor.astNavigator().get
      result = Option.apply(typeOf(value.get, astNode, navigator))
    }
    result
  }

  def typeOf(result: TypeGraph, astNode: AstNode, astNavigator: AstNavigator): WeaveType = {
    result
      .findNode(astNode) match {
        case Some(typenode) => typenode.resultType().orNull
        case None => {
          val mayBeParent = astNavigator.parentOf(astNode)
          mayBeParent match {
            case Some(parentAst) if (parentAst.location().startPosition.index == astNode.location().startPosition.index) => {
              typeOf(result, parentAst, astNavigator)
            }
            case _ => null
          }
        }
      }
  }

  /**
    * Returns the type of the element at that offset
    *
    * @param offset The offset of the element
    * @return The type or null if no type was found
    */
  def typeOf(offset: Int): WeaveType = {
    val value = editor.typeGraph()
    var result: WeaveType = null
    if (value.isDefined) {
      val navigator = editor.astNavigator().get
      val maybeNode = navigator.nodeAt(offset)
      if (maybeNode.isDefined) {
        result = typeOf(value.get, maybeNode.get, navigator)
      }
    }
    result
  }

  /**
    * Returns the type of the expression between that range
    *
    * @param startOffset The start index of the element
    * @param endOffset   The end index of the element
    * @return
    */
  def typeOf(startOffset: Int, endOffset: Int): WeaveType = {
    val value = editor.typeGraph()
    var result: WeaveType = null
    if (value.isDefined) {
      val navigator = editor.astNavigator().get
      val maybeNode = navigator.nodeAt(startOffset, endOffset)
      if (maybeNode.isDefined) {
        result = typeOf(value.get, maybeNode.get, navigator)
      }
    }
    result
  }
}
