package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.MaxTypeGraphExecution
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.ts.WeaveTypeTraverse.equalsWith
import org.mule.weave.v2.ts.resolvers.LiteralTypeResolver

case class TypeNode(astNode: AstNode, typeResolver: WeaveTypeResolver) {

  private val MAX_ATTEMPTS = 15
  private var _outgoingEdges: Seq[Edge] = Seq()
  private var _incomingEdges: Seq[Edge] = Seq()
  private var _incomingType: Option[WeaveType] = None
  private var _originalIncomingType: Option[WeaveType] = None
  private var executionCount: Int = 0
  private var reverseExecutionCount: Int = 0
  private var parentGraph: TypeGraph = _
  private var changeLocation: Boolean = true

  def location(): WeaveLocation = astNode.location()

  def withParent(parentGraph: TypeGraph): TypeNode = {
    this.parentGraph = parentGraph
    this
  }

  def parent(): TypeGraph = {
    parentGraph
  }

  /**
    * Mark this node as internal
    */
  def internalNode(): TypeNode = {
    changeLocation = false
    this
  }

  /**
    * Resolves the result type of the node and propagates it through the outgoing edges
    *
    * @param ctx The resolution context
    */
  def resolve(ctx: WeaveTypeResolutionContext): Unit = {
    ctx.currentParsingContext.notificationManager.progress()
    if (typeResolver.supportsPartialResolution() || allDependenciesResolved()) {
      if (executionCount < MAX_ATTEMPTS) {
        executionCount = executionCount + 1
        //We clear message before executing as messages may be fixed in this execution
        ctx.clearMessageFor(this)
        val resultType: Option[WeaveType] = typeResolver.resolveReturnType(this, ctx)
        if (resultType.isDefined) {
          if (_incomingType.isEmpty || !equalsWith(resultType.get, _originalIncomingType.get)) {
            //We only add one if the type is new
            _originalIncomingType = resultType
            _incomingType = if (!changeLocation) {
              resultType
            } else {
              resultType.map((wt) => {
                //Put the
                WeaveTypeCloneHelper.clone[WeaveType](wt, astNode.location())
              })
            }
            _outgoingEdges.foreach((edge) => {
              val newType: WeaveType = _incomingType.get
              edge.propagateType(newType, ctx)
            })
          }
        }
      } else {
        ctx.warning(MaxTypeGraphExecution("Inference", MAX_ATTEMPTS), this)
      }
    }
  }

  def reverseResolve(ctx: WeaveTypeResolutionContext): Unit = {
    reverseExecutionCount = reverseExecutionCount + 1
    if (reverseExecutionCount < MAX_ATTEMPTS) {
      val expectedType = if (_outgoingEdges.exists(_.expectedPropagatedTypeDefined())) {
        Some(TypeHelper(ctx).unify(_outgoingEdges.flatMap(_.mayBeExpectedPropagatedType())))
      } else {
        None
      }
      typeResolver
        .resolveExpectedType(this, expectedType, ctx)
        .foreach((edgeType) => {
          val edge: Edge = edgeType._1
          val weaveType: WeaveType = edgeType._2
          edge.reversePropagate(weaveType, ctx)
        })
    } else {
      ctx.warning(MaxTypeGraphExecution("Reverse", MAX_ATTEMPTS), this)
    }
  }

  def dispose(): Unit = {
    _outgoingEdges = Seq()
    _incomingEdges.foreach((edge) => {
      edge.source.removeOutputEdge(edge)
    })
    _incomingEdges = Seq()
    parentGraph = null
  }

  def containsEdge(labelName: EdgeLabel): Boolean = {
    _incomingEdges.exists(_.label.forall(_.equals(labelName)))
  }

  def removeOutputEdge(edge: Edge): Unit = {
    _outgoingEdges = _outgoingEdges.filterNot(_ eq edge)
  }

  def removeIncomingEdge(edge: Edge): Unit = {
    _incomingEdges = _incomingEdges.filterNot(_ eq edge)
  }

  def allDependenciesResolved(): Boolean = {
    incomingEdges().isEmpty || typeResolver.isInstanceOf[LiteralTypeResolver] || incomingEdges().forall(_.incomingTypeDefined())
  }

  def resultType(): Option[WeaveType] = {
    _incomingType
  }

  def addOutgoingEdge(edge: Edge): Unit = {
    _outgoingEdges = _outgoingEdges.:+(edge)
  }

  def addIncomingEdge(edge: Edge): Unit = {
    _incomingEdges = _incomingEdges.:+(edge)
  }

  def outgoingEdges(): Seq[Edge] = {
    _outgoingEdges
  }

  def collectExpectedType(): Option[WeaveType] = {
    val weaveTypes = outgoingEdges().flatMap(_.mayBeExpectedPropagatedType())
    if (weaveTypes.isEmpty) {
      None
    } else {
      Some(TypeHelper.unify(weaveTypes))
    }
  }

  def incomingEdges(): Seq[Edge] = {
    _incomingEdges
  }

  def incomingEdges(label: EdgeLabel): Seq[Edge] = {
    _incomingEdges.filter((edge) => edge.label.exists(_.equals(label)))
  }

  def incomingEdge(label: EdgeLabel): Option[Edge] = {
    incomingEdges(label).headOption
  }

  def incomingType(label: EdgeLabel): Option[WeaveType] = {
    incomingEdge(label).map(_.incomingType())
  }

  def incomingTypes(label: EdgeLabel): Seq[WeaveType] = {
    incomingEdges(label).map(_.incomingType())
  }

  def incomingTypes(): Seq[WeaveType] = {
    incomingEdges().map(_.incomingType())
  }

}
