package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.parser.InvalidKeyType
import org.mule.weave.v2.ts.Edge
import org.mule.weave.v2.ts.EdgeLabels
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.NameValuePairType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeCloneHelper.copyLocation
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.WeaveTypeResolver
import org.mule.weave.v2.ts.resolvers.NameTypeResolver.toNameType
import org.mule.weave.v2.ts.resolvers.NameValuePairTypeResolver.selectNameValuePair

import scala.collection.Seq

object KeyTypeResolver extends WeaveTypeResolver {

  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val attrsEdge: Seq[Edge] = attributeEdges(node)
    val attributes: Seq[NameValuePairType] =
      attrsEdge
        .map(_.incomingType())
        .flatMap({
          case name: NameValuePairType => Seq(name)
          case ot: ObjectType =>
            ot.properties.flatMap((kvp) => {
              kvp.key match {
                case keyType: KeyType => {
                  Some(NameValuePairType(keyType.name, kvp.value))
                }
                case _ => Seq()
              }
            })
          case _ => Seq()
        })
    val localName = node.incomingEdge(EdgeLabels.LOCAL_NAME)
    val ns = node.incomingEdge(EdgeLabels.NAMESPACE)
    val namespaceType: Option[WeaveType] = ns.map(_.incomingType())
    val localNameType: WeaveType = localName.get.incomingType()
    localNameType match {
      case keyType: KeyType if attributes.isEmpty =>
        Some(keyType)
      case _ =>
        val maybeQName = toNameType(localNameType, namespaceType)
        maybeQName match {
          case Some(qName) => {
            copyLocation(qName, node.location())
            Some(KeyType(qName, attributes))
          }
          case None => {
            ctx.error(InvalidKeyType(localNameType), node)
            None
          }
        }

    }
  }

  override def resolveExpectedType(node: TypeNode, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): Seq[(Edge, WeaveType)] = {
    incomingExpectedType match {
      case Some(weaveType: KeyType) => {
        attributeEdges(node).flatMap((attrEdge) => {
          val maybeNameValuePairType = selectNameValuePair(attrEdge.source, weaveType)
          maybeNameValuePairType.map((nvp) => (attrEdge, nvp))
        })
      }
      case _ => Seq()
    }
  }

  private def attributeEdges(node: TypeNode) = {
    node.incomingEdges(EdgeLabels.ATTRIBUTE)
  }
}
