package org.mule.weave.v2.utils

import org.mule.weave.v2.parser.annotation.WithValueAstNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.CommentNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.WeaveLocationCapable
import org.mule.weave.v2.parser.ast.header.directives.VersionDirective
import org.mule.weave.v2.parser.ast.operators.OpNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.Position

class AstEmitter(printLocation: Boolean = true, printComments: Boolean = false, printAnnotations: Boolean = false) {

  def print(document: AstNode): String = {
    val builder: StringBuilder = new StringBuilder
    print(builder, document, 0)
    builder.toString()
  }

  private def print(builder: StringBuilder, node: AstNode, indent: Int): Unit = {
    printName(builder, node, indent)
    node match {
      case capable: WeaveLocationCapable =>
        if (printLocation) {
          val startPosition: Position = capable.location().startPosition
          if (startPosition != null) {
            builder.append(s" Start='${startPosition.line},${startPosition.column},${startPosition.index}'")
          }
          val endPosition: Position = capable.location().endPosition
          if (endPosition != null) {
            builder.append(s" End='${endPosition.line},${endPosition.column},${endPosition.index}'")
          }
        }
        capable match {
          case lv: LiteralValueAstNode => {
            builder.append(s" Value='${lv.literalValue}'")
            lv match {
              case cn: CommentNode => builder.append(s" Type='${cn.commentType.toString}'")
              case _               =>
            }
          }
          case uen: UpdateExpressionNode => builder.append(s"ForceCreate=${uen.forceCreate}")
          case opNode: OpNode =>
            builder.append(s" Name='${opNode.opId.name}'")
          case ns: NameIdentifier =>
            builder.append(" Name='").append(ns.name).append("'")
          case vd: VersionDirective =>
            builder.append(s" Minor='${vd.minor.v}'").append(s" Major='${vd.major.v}'")
          case _ =>
        }
      case _ =>
    }
    if (printAnnotations) {
      node
        .annotations()
        .foreach((an) => {
          builder.append(" @" + an.name())
          an match {
            case vasn: WithValueAstNodeAnnotation[_] => builder.append("(").append(vasn.value()).append(")")
            case _                                   =>
          }
        })
    }
    builder.append("\n")
    if (!node.isInstanceOf[VersionDirective]) {
      if (printComments) {
        node.comments.foreach((item: AstNode) => {
          print(builder, item, indent + 1)
        })
      }
      node
        .children()
        .foreach((item: AstNode) => {
          print(builder, item, indent + 1)
        })
    }
  }

  private def printName(builder: StringBuilder, astNode: AstNode, indent: Int): Unit = {
    val indentString: String = "  ".*(indent - 1) + "|-"
    builder.append(indentString).append(astNode.getClass.getSimpleName)
  }

}

object AstEmitter {
  def apply(): AstEmitter = {
    new AstEmitter()
  }

  def apply(printLocation: Boolean, printComments: Boolean): AstEmitter = {
    new AstEmitter(printLocation, printComments)
  }

  def apply(printLocation: Boolean, printComments: Boolean, printAnnotations: Boolean = false): AstEmitter = {
    new AstEmitter(printLocation, printComments, printAnnotations)
  }

}
