package org.mule.weave.v2.versioncheck

import org.mule.weave.v2.grammar.MetadataAdditionOpId
import org.mule.weave.v2.grammar.MetadataInjectorOpId
import org.mule.weave.v2.parser.IncompatibleRuntimeVersion
import org.mule.weave.v2.parser.LanguageFeatureNotAvailable
import org.mule.weave.v2.parser.annotation.SinceAstNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.DirectivesCapableNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationCapableNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.types.LiteralTypeNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.types.TypeSelectorNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNodeWithSchema
import org.mule.weave.v2.parser.ast.updates.UpdateNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.CompilationPhase
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware
import org.mule.weave.v2.parser.phase.SuccessResult

class LanguageLevelVersionCheckPhase[R <: AstNode, T <: AstNodeResultAware[R] with ScopeNavigatorResultAware]() extends CompilationPhase[T, T] {

  override def doCall(source: T, context: ParsingContext): PhaseResult[_ <: T] = {
    val nameIdentifier = context.nameIdentifier
    if (context.languageLevel.isDefined && !isDataWeaveSdkFile(nameIdentifier)) {
      val maxVersion = context.languageLevel.get

      val variableReferenceNodes = AstNodeHelper.collectChildrenWith(source.astNode, classOf[VariableReferenceNode])
      variableReferenceNodes.foreach((vrn) => {
        val variable = vrn.variable
        checkRefCorrectVersion(source, variable, maxVersion, context, vrn)
      })

      val typeReferenceNode = AstNodeHelper.collectChildrenWith(source.astNode, classOf[TypeReferenceNode])
      typeReferenceNode.foreach((vrn) => {
        val variable = vrn.variable
        checkRefCorrectVersion(source, variable, maxVersion, context, vrn)
      })

      val namespaces = AstNodeHelper.collectChildrenWith(source.astNode, classOf[NamespaceNode])
      namespaces.foreach((vrn) => {
        val variable = vrn.prefix
        checkRefCorrectVersion(source, variable, maxVersion, context, vrn)
      })

      if (SVersion._230 > maxVersion) {
        check230LanguageFeatures(source, context, maxVersion);
      }

      if (SVersion._250 > maxVersion) {
        check250LanguageFeatures(source, context, maxVersion);
      }

      if (SVersion._280 > maxVersion) {
        check280languageFeatures(source, context, maxVersion);
      }

    }
    SuccessResult(source, context)
  }

  def check230LanguageFeatures(source: T, context: ParsingContext, currentVersion: SVersion): Unit = {
    checkDataFormatID(source, context, currentVersion)

    val updateNodes = AstNodeHelper.collectChildrenWith(source.astNode, classOf[UpdateNode])
    updateNodes.foreach((un) => {
      val errorMessage = LanguageFeatureNotAvailable(currentVersion.semVerString(), SVersion._230.semVerString(), "Update Operator")
      context.messageCollector.error(errorMessage, un.location())
    })

    val literalTypes = AstNodeHelper.collectChildrenWith(source.astNode, classOf[LiteralTypeNode])
    literalTypes.foreach((un) => {
      val errorMessage = LanguageFeatureNotAvailable(currentVersion.semVerString(), SVersion._230.semVerString(), "Literal Type")
      context.messageCollector.error(errorMessage, un.location())
    })
  }

  private def checkDataFormatID(source: T, context: ParsingContext, currentVersion: SVersion): Unit = {
    source.astNode match {
      case directivesCapableNode: DirectivesCapableNode => {
        AstNodeHelper
          .getOutputDirective(directivesCapableNode)
          .foreach((od) => {
            if (od.dataFormat.isDefined) {
              val errorMessage = LanguageFeatureNotAvailable(currentVersion.semVerString(), SVersion._230.semVerString(), "Data Format by ID")
              context.messageCollector.error(errorMessage, od.location())
            }
          })
      }
      case _ =>
    }
  }

  private def checkRefCorrectVersion(source: T, variable: NameIdentifier, maxVersion: SVersion, context: ParsingContext, astNode: AstNode): Unit = {
    val maybeReference = source.scope.resolveVariable(variable)
    maybeReference.foreach((ref) => {
      ref.moduleSource match {
        case Some(module) if (isDataWeaveSdkFile(module)) => {
          val maybeNode = ref.scope.astNavigator().parentWithType(ref.referencedNode, classOf[AnnotationCapableNode])
          maybeNode.foreach((annotationCapableNode) => {
            val maybeSince = annotationCapableNode.annotation(classOf[SinceAstNodeAnnotation])
            maybeSince match {
              case Some(sinceAnnotation) => {
                val version = sinceAnnotation.version
                if (!(maxVersion >= version)) {
                  val errorMessage = IncompatibleRuntimeVersion(maxVersion.semVerString(), version.semVerString(), ref.referencedNode)
                  context.messageCollector.error(errorMessage, astNode.location())
                }
              }
              case _ =>
            }
          })
        }
        case _ =>
      }
    })
  }

  def check250LanguageFeatures(source: T, context: ParsingContext, maxVersion: SVersion) = {
    val typeSelectorNodes = AstNodeHelper.collectChildrenWith(source.astNode, classOf[TypeSelectorNode])
    typeSelectorNodes.foreach(typeSelectorNode => {
      val errorMessage = LanguageFeatureNotAvailable(maxVersion.semVerString(), SVersion._250.semVerString(), "Type Selector")
      context.messageCollector.error(errorMessage, typeSelectorNode.location())
    })
    val metadataInjectorNodes = AstNodeHelper
      .collectChildrenWith(source.astNode, classOf[BinaryOpNode])
      .filter(n =>
        n match {
          case BinaryOpNode(MetadataInjectorOpId, _, _, _) => true
          case _ => false
        })
    metadataInjectorNodes.foreach(metadataInjectorNode => {
      val errorMessage = LanguageFeatureNotAvailable(maxVersion.semVerString(), SVersion._250.semVerString(), "Metadata Assignment")
      context.messageCollector.error(errorMessage, metadataInjectorNode.location())
    })

    val typeMetadataNodes = AstNodeHelper.collectChildrenWith(source.astNode, classOf[WeaveTypeNodeWithSchema]).filter(_.asTypeSchema.isDefined)
    typeMetadataNodes.foreach(node => {
      val errorMessage = LanguageFeatureNotAvailable(maxVersion.semVerString(), SVersion._250.semVerString(), "Type Metadata Assignment")
      context.messageCollector.error(errorMessage, node.location())
    })
  }

  def check280languageFeatures(source: T, context: ParsingContext, maxVersion: SVersion) = {
    val metadataAdditionNodes = AstNodeHelper
      .collectChildrenWith(source.astNode, classOf[BinaryOpNode])
      .filter(n =>
        n match {
          case BinaryOpNode(MetadataAdditionOpId, _, _, _) => true
          case _ => false
        })
    metadataAdditionNodes.foreach(metadataInjectorNode => {
      val errorMessage = LanguageFeatureNotAvailable(maxVersion.semVerString(), SVersion._280.semVerString(), "Metadata annotations in expressions")
      context.messageCollector.error(errorMessage, metadataInjectorNode.location())
    })
    val metadataAnnotations = AstNodeHelper
      .collectChildrenWith(source.astNode, classOf[AnnotationDirectiveNode])
      .filter(n =>
        n match {
          case AnnotationDirectiveNode(_, _, annotationNodes) if annotationNodes.exists(p => isMetadataAnnotation(source, p)) => true
          case _ => false
        })
    metadataAnnotations.foreach(metadataAnnotation => {
      val errorMessage = LanguageFeatureNotAvailable(maxVersion.semVerString(), SVersion._280.semVerString(), "Metadata annotations directives")
      context.messageCollector.error(errorMessage, metadataAnnotation.location())
    })
  }

  private def isMetadataAnnotation(source: T, p: AnnotationNode) = {
    source.scope.resolveVariable(p.name).exists(p => NameIdentifier.METADATA_ANNOTATION.equals(p.fqnReferenceName))
  }

  private def isDataWeaveSdkFile(nameIdentifier: NameIdentifier) = {
    nameIdentifier.name.startsWith("dw::")
  }
}
