package org.mule.weave.v2.editor.indexing

import org.mule.weave.v2.editor.ChangeListener
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ParsingContext

import java.util
import java.util.Collections
import scala.collection.JavaConverters.asJavaIteratorConverter
import scala.collection.mutable

/**
  * This service allows the Tooling to query for different things that has been indexed
  */
trait WeaveIndexService {

  /**
    * Searches all Global Accessible Symbols by name
    * @param name The name of the element to search
    * @return All the elements that matched
    */
  def searchDefinitions(name: String): Array[LocatedResult[WeaveIdentifier]]

  /**
    * Searches all references that name
    * @param name The name of the reference being searched
    * @return All The NameIdentifiers that has that name
    */
  def searchReferences(name: String): Array[LocatedResult[WeaveIdentifier]]

  /**
    * Searches all references that name
    * @param name The name of the reference being searched
    * @return All The NameIdentifiers that has that name
    */
  def searchDocumentContainingName(name: String): java.util.Iterator[LocatedResult[WeaveDocument]]

}
object EmptyIndexService extends WeaveIndexService {
  override def searchDefinitions(name: String): Array[LocatedResult[WeaveIdentifier]] = Array.empty

  override def searchReferences(name: String): Array[LocatedResult[WeaveIdentifier]] = Array.empty

  override def searchDocumentContainingName(name: String): util.Iterator[LocatedResult[WeaveDocument]] = Collections.emptyIterator()
}

class SimpleWeaveIndexService(vfs: VirtualFileSystem, newParsingContext: (NameIdentifier) => ParsingContext) extends WeaveIndexService {

  private var initialized: Boolean = false
  private val indexedIdentifiers: mutable.HashMap[String, Array[LocatedResult[WeaveIdentifier]]] = mutable.HashMap()
  private val indexedNames: mutable.HashMap[String, LocatedResult[WeaveDocument]] = mutable.HashMap()

  private def indexProjectFile(vf: VirtualFile) = {
    val indexer: DefaultWeaveIndexer = new DefaultWeaveIndexer()
    val nameIdentifier: NameIdentifier = vf.getNameIdentifier
    if (indexer.parse(vf, newParsingContext(nameIdentifier))) {
      indexedIdentifiers.put(vf.url(), index(vf, indexer))
      indexedNames.put(vf.url(), LocatedResult[WeaveDocument](nameIdentifier, indexer.document()))
    }
  }

  private def index(vf: VirtualFile, indexer: DefaultWeaveIndexer): Array[LocatedResult[WeaveIdentifier]] = {
    val nameIdentifier = vf.getNameIdentifier
    val weaveSymbols: Array[WeaveIdentifier] = indexer.identifiers()
    val locatedWeaveSymbols = weaveSymbols.map((symbol) => {
      LocatedResult[WeaveIdentifier](nameIdentifier, symbol)
    })
    locatedWeaveSymbols
  }

  def init(): Unit = {
    if (!initialized) {
      registerListener()
      val value: util.Iterator[VirtualFile] = vfs.listFiles()
      while (value.hasNext) {
        indexProjectFile(value.next())
      }
    }
    initialized = true
  }

  private def registerListener() = {
    vfs.changeListener(new ChangeListener {
      override def onDeleted(vf: VirtualFile): Unit = {
        indexedIdentifiers.remove(vf.url())
        indexedNames.remove(vf.url())
      }

      override def onChanged(vf: VirtualFile): Unit = {
        indexProjectFile(vf)
      }

      override def onCreated(vf: VirtualFile): Unit = {
        indexProjectFile(vf)
      }
    })
  }

  private def searchIn(identifiers: Iterable[Array[LocatedResult[WeaveIdentifier]]], name: String, kind: Int): Iterable[LocatedResult[WeaveIdentifier]] = {
    identifiers
      .flatMap((identifiers) => {
        identifiers.filter((identifier) => {
          identifier.value.kind == kind &&
            identifier.value.value.equals(name)
        })
      })
  }

  override def searchDefinitions(name: String): Array[LocatedResult[WeaveIdentifier]] = {
    init()
    searchIn(indexedIdentifiers.values, name, IdentifierKind.DEFINITION).toArray
  }

  override def searchReferences(name: String): Array[LocatedResult[WeaveIdentifier]] = {
    init()
    searchIn(indexedIdentifiers.values, name, IdentifierKind.REFERENCE).toArray
  }

  override def searchDocumentContainingName(name: String): util.Iterator[LocatedResult[WeaveDocument]] = {
    init()
    indexedNames.values.toIterator
      .filter((indexedName) => {
        indexedName.moduleName.name.contains(name)
      })
      .asJava
  }
}
