package org.mule.weave.v2.inspector

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.grammar.EqOpId
import org.mule.weave.v2.grammar.IsOpId
import org.mule.weave.v2.grammar.NotEqOpId
import org.mule.weave.v2.parser.UsingIfNotNullInsteadOfDefault
import org.mule.weave.v2.parser.UsingIfNullInsteadOfDefault
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.conditional.IfNode
import org.mule.weave.v2.parser.ast.functions.DoBlockNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.NullNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware

object IfNotNullInspector extends CodeInspector[AstNodeResultAware[_] with ScopeNavigatorResultAware] {
  override def inspect(node: AstNode, scopeData: AstNodeResultAware[_] with ScopeNavigatorResultAware, parsingContext: ParsingContext): Unit = {
    node match {
      case ifnode @ IfNode(ifExpr, BinaryOpNode(NotEqOpId, lhs, _: NullNode, _), elseExpr, _) => {
        //if(p.a != null) p.a else "test"
        val actualExpression = resolveExpression(ifExpr)
        if (actualExpression.equals(lhs)) {
          parsingContext.messageCollector.warning(UsingIfNotNullInsteadOfDefault(lhs, elseExpr, ifnode), ifnode.location())
        }
      }
      case ifnode @ IfNode(ifExpr, BinaryOpNode(EqOpId, lhs, _: NullNode, _), elseExpr, _) => {
        //if(p.a == null) "test" else p.a
        val actualExpression = resolveExpression(elseExpr)
        if (actualExpression.equals(lhs)) {
          parsingContext.messageCollector.warning(UsingIfNullInsteadOfDefault(lhs, ifExpr, ifnode), ifnode.location())
        }
      }
      case ifnode @ IfNode(ifExpr, BinaryOpNode(IsOpId, lhs, TypeReferenceNode(NameIdentifier("Null", None), None, None, None, _), _), elseExpr, _) => {
        //if(p.a is Null) "test" else p.a
        val actualExpression = resolveExpression(elseExpr)
        if (actualExpression.equals(lhs)) {
          parsingContext.messageCollector.warning(UsingIfNullInsteadOfDefault(lhs, ifExpr, ifnode), ifnode.location())
        }
      }
      case _ =>
    }
  }

  private def resolveExpression(ifExpr: AstNode): AstNode = {
    ifExpr match {
      case doBlock: DoBlockNode if (doBlock.header.directives.isEmpty) => {
        resolveExpression(doBlock.body)
      }
      case expr => expr
    }
  }
}

class IfNotNullQuickFixAction(expression: AstNode, defaultValue: AstNode, ifExpression: AstNode) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {
    val startIndex = ifExpression.location().startPosition.index
    document.delete(startIndex, ifExpression.location().endPosition.index)
    document.insert(CodeGenerator.generate(expression) + " default " + CodeGenerator.generate(defaultValue), startIndex)
  }
}
