package org.mule.weave.v2.parser.ast.types

import org.mule.weave.v2.api.tooling.ast.DWAstNodeKind
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.Child
import org.mule.weave.v2.parser.ast.Children
import org.mule.weave.v2.parser.ast.annotation.AnnotationCapableNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

sealed trait WeaveTypeNode extends AstNode {
  override def cloneAst(): WeaveTypeNode = {
    super.cloneAst().asInstanceOf[WeaveTypeNode]
  }
}

sealed trait WeaveTypeNodeWithSchema extends WeaveTypeNode with AnnotationCapableNode {

  def asSchema: Option[SchemaNode]

  def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema

  def asTypeSchema: Option[SchemaNode]

  def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema
}

case class ObjectTypeNode(var properties: Seq[WeaveTypeNode], var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, close: Boolean = false, ordered: Boolean = false, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {

  override def children(): Seq[AstNode] = Children().++=(properties).+=(asSchema).+=(asTypeSchema).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNodeWithSchema = this.copy(properties.map(_.cloneAst()))

  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def getKind(): String = DWAstNodeKind.OBJECT_TYPE_NODE
}

case class KeyValueTypeNode(var key: WeaveTypeNode, var value: WeaveTypeNode, repeated: Boolean, optional: Boolean) extends WeaveTypeNode {
  override def children(): Seq[AstNode] = Child(key, value)

  protected override def doClone(): WeaveTypeNode = this.copy(key.cloneAst(), value.cloneAst())

  override def getKind(): String = DWAstNodeKind.KEY_VALUE_PAIR_TYPE_NODE
}

case class NameValueTypeNode(var name: NameTypeNode, var value: WeaveTypeNode, optional: Boolean) extends WeaveTypeNode {
  override def children(): Seq[AstNode] = Child(name, value)

  protected override def doClone(): WeaveTypeNode = this.copy(name.cloneAst().asInstanceOf[NameTypeNode], value.cloneAst())

  override def getKind(): String = DWAstNodeKind.NAME_VALUE_TYPE_NODE
}

case class KeyTypeNode(var name: NameTypeNode, var attrs: Seq[WeaveTypeNode] = Seq(), var codeAnnotations: Seq[AnnotationNode] = Seq(), var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None) extends WeaveTypeNodeWithSchema {
  override def children(): Seq[AstNode] = Children(name).++=(attrs).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNode = this.copy(name.cloneAst().asInstanceOf[NameTypeNode], attrs.map(_.cloneAst()), codeAnnotations.map(_.cloneAst()))

  /**
    * Sets the annotations to this node
    *
    * @param annotations The list of annotations
    */
  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def getKind(): String = DWAstNodeKind.KEY_TYPE_NODE
}

case class TypeParameterNode(var name: NameIdentifier, var base: Option[WeaveTypeNode]) extends WeaveTypeNode {
  override def children(): Seq[AstNode] = Children(name).+=(base).result()

  protected override def doClone(): WeaveTypeNode = this.copy()

  override def cloneAst(): TypeParameterNode = super.cloneAst().asInstanceOf[TypeParameterNode]

  override def getKind(): String = DWAstNodeKind.TYPE_PARAMETER_NODE
}

case class FunctionTypeNode(var args: Seq[FunctionParameterTypeNode], var returnType: WeaveTypeNode, var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {
  override def children(): Seq[AstNode] = Children().++=(args).+=(returnType).+=(asSchema).+=(asTypeSchema).++=(codeAnnotations).result()
  protected override def doClone(): WeaveTypeNodeWithSchema = this.copy()
  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def getKind(): String = DWAstNodeKind.FUNCTION_TYPE_NODE
}

case class FunctionParameterTypeNode(name: Option[NameIdentifier], valueType: WeaveTypeNode, optional: Boolean) extends WeaveTypeNode {
  override def children(): Seq[AstNode] = Children().+=(name).+=(valueType).result()

  protected override def doClone(): WeaveTypeNode = this.copy()

  override def getKind(): String = DWAstNodeKind.FUNCTION_PARAMETER_TYPE_NODE
}

case class UnionTypeNode(var of: Seq[WeaveTypeNode], var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {
  override def children(): Seq[AstNode] = Children().++=(of).+=(asSchema).+=(asTypeSchema).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNodeWithSchema = this.copy(of.map(_.cloneAst()), asSchema, maybeTypeSchema, codeAnnotations.map(_.cloneAst()))

  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def getKind(): String = DWAstNodeKind.UNION_TYPE_NODE
}

case class IntersectionTypeNode(var of: Seq[WeaveTypeNode], var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {
  override def children(): Seq[AstNode] = Children().++=(of).+=(asSchema).+=(maybeTypeSchema).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNodeWithSchema = this.copy(of.map(_.cloneAst()), asSchema)

  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def getKind(): String = DWAstNodeKind.INTERSECTION_TYPE_NODE
}

case class NativeTypeNode(var typeId: String, asSchema: Option[SchemaNode] = None) extends WeaveTypeNode {

  override def children(): Seq[AstNode] = Child(asSchema)

  protected override def doClone(): WeaveTypeNode = this.copy()

  override def getKind(): String = DWAstNodeKind.NATIVE_TYPE_NODE
}

case class TypeReferenceNode(variable: NameIdentifier, var typeArguments: Option[Seq[WeaveTypeNode]] = None, var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {
  override def children(): Seq[AstNode] = Children(variable).++=(typeArguments).+=(asSchema).+=(asTypeSchema).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNodeWithSchema = this.copy()

  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def getKind(): String = DWAstNodeKind.TYPE_REFERENCE_NODE
}

case class TypeSelectorNode(selector: NameNode, var weaveTypeNode: WeaveTypeNode, var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {

  override def children(): Seq[AstNode] = Children(selector).+=(weaveTypeNode).+=(asSchema).+=(asTypeSchema).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNodeWithSchema = this.copy()

  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def getKind(): String = DWAstNodeKind.TYPE_SELECTOR_NODE
}

case class NameTypeNode(localName: Option[String], ns: Option[NamespaceNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {
  override def children(): Seq[AstNode] = Children().++=(ns.toSeq).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNode = this.copy()

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def asSchema: Option[SchemaNode] = None

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = this

  override def getKind(): String = DWAstNodeKind.NAME_TYPE_NODE
}

case class DynamicReturnTypeNode() extends WeaveTypeNode {
  protected override def doClone(): WeaveTypeNode = DynamicReturnTypeNode()

  override def getKind(): String = DWAstNodeKind.DYNAMIC_RETURN_TYPE_NODE
}

case class LiteralTypeNode(value: AstNode, var maybeSchema: Option[SchemaNode] = None, var maybeTypeSchema: Option[SchemaNode] = None, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends WeaveTypeNodeWithSchema {

  override def children(): Seq[AstNode] = Children(value).+=(asSchema).+=(asTypeSchema).++=(codeAnnotations).result()

  protected override def doClone(): WeaveTypeNode = this.copy()

  override def asSchema: Option[SchemaNode] = maybeSchema

  override def withSchema(schema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeSchema = Some(schema)
    this
  }

  override def asTypeSchema: Option[SchemaNode] = maybeTypeSchema

  override def withTypeSchema(typeSchema: SchemaNode): WeaveTypeNodeWithSchema = {
    maybeTypeSchema = Some(typeSchema)
    this
  }

  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = this.codeAnnotations = annotations

  override def getKind(): String = DWAstNodeKind.LITERAL_TYPE_NODE
}
