package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.parser.InvalidNameType
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.EdgeLabels
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NamespaceType
import org.mule.weave.v2.ts.NothingType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.UriType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.WeaveTypeResolver

object NameTypeResolver extends WeaveTypeResolver {
  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val localName = node.incomingEdge(EdgeLabels.LOCAL_NAME)
    val ns = node.incomingEdge(EdgeLabels.NAMESPACE)
    val namespaceType: Option[WeaveType] = ns.map(_.incomingType())
    val localNameType: WeaveType = localName.get.incomingType()
    val qName = toNameType(localNameType, namespaceType)
    qName match {
      case Some(qName) => Some(qName)
      case None => {
        ctx.error(InvalidNameType(localNameType), node)
        None
      }
    }
  }

  def toNameType(localNameType: WeaveType, namespaceType: Option[WeaveType]): Option[NameType] = {
    localNameType match {
      case ut: UnionType if ut.of.forall(toNameType(_, namespaceType).isDefined) => Some(NameType())
      case kt: KeyType => toNameType(kt.name, namespaceType)
      case nt @ NameType(value) => {
        namespaceType match {
          case Some(NamespaceType(_, Some(UriType(Some(nsValue))))) if value.isDefined => Some(NameType(Some(QName(value.get.name, Some(nsValue)))))
          case _ => Some(nt)
        }
      }
      case StringType(Some(localNameValue)) => {
        namespaceType match {
          case Some(NamespaceType(_, Some(UriType(Some(nsValue))))) => Some(NameType(Some(QName(localNameValue, Some(nsValue)))))
          case _ => Some(NameType(Some(QName(localNameValue))))
        }
      }
      case StringType(None) => Some(NameType())
      case NumberType(Some(value)) => Some(NameType(Some(QName(value))))
      case NumberType(None) => Some(NameType())
      case BooleanType(Some(value), _) => Some(NameType(Some(QName(value.toString))))
      case BooleanType(None, _) => Some(NameType())
      case TypeParameter(_, None, _, _, _) => Some(NameType())
      case TypeParameter(_, Some(top), _, _, _) if toNameType(top, namespaceType).isDefined => Some(NameType())
      case rt: ReferenceType => toNameType(rt.resolveType(), namespaceType)
      case _: NothingType => Some(NameType())
      case _ => None
    }
  }
}
