package org.mule.weave.v2.editor.composer

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.ValidationMessage
import org.mule.weave.v2.grammar.AdditionOpId
import org.mule.weave.v2.grammar.AsOpId
import org.mule.weave.v2.grammar.DivisionOpId
import org.mule.weave.v2.grammar.MinusOpId
import org.mule.weave.v2.grammar.MultiplicationOpId
import org.mule.weave.v2.grammar.NotOpId
import org.mule.weave.v2.grammar.SubtractionOpId
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.WeaveLocationCapable
import org.mule.weave.v2.parser.ast.conditional.DefaultNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.logical.AndNode
import org.mule.weave.v2.parser.ast.logical.OrNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.operators.UnaryOpNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.DateTimeNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.LocalDateNode
import org.mule.weave.v2.parser.ast.structure.LocalDateTimeNode
import org.mule.weave.v2.parser.ast.structure.LocalTimeNode
import org.mule.weave.v2.parser.ast.structure.NullNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.StringInterpolationNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.TimeNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.TypeCheckingResult
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.utils.StringEscapeHelper

class ComposerExpressionParser extends ComposerExpressionValidator {

  def parseExpression(expression: String, name: String, parsingContext: ParsingContext): Option[ComposerExpressionNode] = {
    val value: PhaseResult[ParsingResult[DocumentNode]] = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource(name, expression), parsingContext)
    if (value.hasResult()) {
      val theResult = value.getResult()
      Some(generateExpressionNode(theResult.astNode.root))
    } else {
      None
    }
  }

  @Deprecated
  def tokenizeTextExpression(expression: String, name: String, parsingContext: ParsingContext): Option[ComposerExpression] = {
    val value: PhaseResult[ParsingResult[DocumentNode]] = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource(name, expression), parsingContext)
    if (value.hasResult()) {
      val theResult = value.getResult()
      Some(ComposerExpression(generateModel(theResult.astNode.root)))
    } else {
      None
    }
  }

  def validate(expression: String, name: String, parsingContext: ParsingContext, allowedFunctions: Seq[String] = Seq.empty): ComposerValidationMessages = {
    val value: PhaseResult[TypeCheckingResult[DocumentNode]] = MappingParser.parse(MappingParser.typeCheckPhase(), WeaveResource(name, expression), parsingContext)
    validateComposerExpression(value, parsingContext, allowedFunctions)
  }

  @Deprecated
  def generate(elements: Seq[ComposerExpressionToken], parsingContext: ParsingContext): String = {
    if (elements.isEmpty) {
      ""
    } else {
      val astNodes = elements.map(element => {
        element.kind match {
          case ComposerExpressionTokenKind.TEXT =>
            val node = StringNode(StringEscapeHelper.escapeString(element.text, '\"', insertQuotes = false))
            if (elements.size == 1)
              node.withQuotation('\"')
            node

          case _ =>
            val expressionText: String = element.text
            val value: PhaseResult[ParsingResult[DocumentNode]] = MappingParser.parse(MappingParser.parsingPhase(), WeaveResource("expression", expressionText), parsingContext)
            value.getResult().astNode.root
        }
      })
      if (astNodes.size > 1) {
        CodeGenerator.generate(StringInterpolationNode(astNodes))
      } else {
        CodeGenerator.generate(astNodes.head)
      }
    }
  }

  @Deprecated
  private def generateModel(astNode: AstNode): Seq[ComposerExpressionNode] = {
    astNode match {
      case lvn: StringNode =>
        val expression: String = StringEscapeHelper.unescapeString(lvn.literalValue, lvn.quotedBy().getOrElse('\"'))
        val part: ComposerExpressionToken = ComposerExpressionToken(ComposerExpressionTokenKind.TEXT, expression)
        Seq(ComposerExpressionNode(part, astNode.location()))

      case si: StringInterpolationNode =>
        si.elements.flatMap(nd => {
          generateModel(nd)
        })

      case en =>
        val expressionPart: ComposerExpressionToken = ComposerExpressionToken(ComposerExpressionTokenKind.PILL, CodeGenerator.generate(en))
        val composerNode: ComposerExpressionNode = ComposerExpressionNode(expressionPart, astNode.location())
        Seq(composerNode)
    }
  }

  private def generateExpressionNode(astNode: AstNode): ComposerExpressionNode = {
    val expressionNode = astNode match {
      case nn: NumberNode =>
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.NUMBER, nn.literalValue)
        ComposerExpressionNode(token, nn.location())

      case bn: BooleanNode =>
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.BOOLEAN, bn.literalValue)
        ComposerExpressionNode(token, bn.location())

      case sn: StringNode =>
        val expression = StringEscapeHelper.unescapeString(sn.literalValue, sn.quotedBy().getOrElse('\"'))
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.TEXT, expression)
        ComposerExpressionNode(token, sn.location())

      case ldn: LocalDateNode =>
        val text = CodeGenerator.generate(ldn)
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.DATE, text)
        ComposerExpressionNode(token, ldn.location())
      case dtn: DateTimeNode =>
        val text = CodeGenerator.generate(dtn)
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.DATE_TIME, text)
        ComposerExpressionNode(token, dtn.location())
      case tn: TimeNode =>
        val text = CodeGenerator.generate(tn)
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.TIME, text)
        ComposerExpressionNode(token, tn.location())
      case ltn: LocalTimeNode =>
        val text = CodeGenerator.generate(ltn)
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.LOCAL_TIME, text)
        ComposerExpressionNode(token, ltn.location())
      case ldtn: LocalDateTimeNode =>
        val text = CodeGenerator.generate(ldtn)
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.LOCAL_DATE_TIME, text)
        ComposerExpressionNode(token, ldtn.location())

      case nulln: NullNode =>
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.NULL)
        ComposerExpressionNode(token, nulln.location())

      case si: StringInterpolationNode =>
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.TEXT_INTERPOLATION, null)
        val children = si.elements.map(nd => generateExpressionNode(nd))
        ComposerExpressionNode(token, si.location(), children.toArray)

      case arrn: ArrayNode =>
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.ARRAY, null)
        val children = arrn.children().map(generateExpressionNode)
        ComposerExpressionNode(token, arrn.location(), children.toArray)

      case typen: TypeReferenceNode =>
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.TYPE, typen.variable.name)
        ComposerExpressionNode(token, typen.location())

      case dn: DefaultNode =>
        val children = Seq(generateExpressionNode(dn.lhs), generateExpressionNode(dn.rhs))
        namedExpressionNode(ComposerExpressionTokenKind.BINARY_OP, "default", dn, children)

      case bon: BinaryOpNode if ComposerExpressionParser.supportedBinaryOpNodeIds(bon.binaryOpId) =>
        val children = Seq(generateExpressionNode(bon.lhs), generateExpressionNode(bon.rhs))
        namedExpressionNode(ComposerExpressionTokenKind.BINARY_OP, bon.binaryOpId.name, bon, children)

      case uon: UnaryOpNode if uon.opId == NotOpId | uon.opId == MinusOpId =>
        val children = Seq(generateExpressionNode(uon.rhs))
        namedExpressionNode(ComposerExpressionTokenKind.UNARY_OP, uon.opId.name, uon, children)

      case an: AndNode =>
        val children = Seq(generateExpressionNode(an.lhs), generateExpressionNode(an.rhs))
        namedExpressionNode(ComposerExpressionTokenKind.FUNCTION_CALL, "and", an, children)

      case or: OrNode =>
        val children = Seq(generateExpressionNode(or.lhs), generateExpressionNode(or.rhs))
        namedExpressionNode(ComposerExpressionTokenKind.FUNCTION_CALL, "or", or, children)

      case fcn @ FunctionCallNode(vrn: VariableReferenceNode, args: FunctionCallParametersNode, _, _) =>
        val children = args.args.map(child => generateExpressionNode(child))
        namedExpressionNode(ComposerExpressionTokenKind.FUNCTION_CALL, vrn.variable.name, fcn, children)

      case en =>
        val pillCode = CodeGenerator.generate(en)
        val parenthesis = en.annotationsBy(classOf[EnclosedMarkAnnotation]).size
        val expressionPart = if (parenthesis != 0) {
          val pillCodeWithoutParenthesis = pillCode.substring(parenthesis, pillCode.length - parenthesis)
          ComposerExpressionToken(ComposerExpressionTokenKind.PILL, pillCodeWithoutParenthesis)
        } else {
          ComposerExpressionToken(ComposerExpressionTokenKind.PILL, pillCode)
        }

        ComposerExpressionNode(expressionPart, en.location())
    }
    wrapEnclosedMarkAnnotation(astNode, expressionNode)
  }

  private def wrapEnclosedMarkAnnotation(astNode: AstNode, expressionNode: ComposerExpressionNode): ComposerExpressionNode = {
    // Using reverse sorting because we need to wrap from the leaf to the root
    val enclosedMarkAnnotations = astNode.annotationsBy(classOf[EnclosedMarkAnnotation]).sortBy(_.location.startPosition.index)(Ordering[Int].reverse)
    if (enclosedMarkAnnotations.nonEmpty) {
      var enclosedExpressionNode = expressionNode
      for (ema <- enclosedMarkAnnotations) {
        val token = ComposerExpressionToken(ComposerExpressionTokenKind.PARENTHESIS, null)
        enclosedExpressionNode = ComposerExpressionNode(token, ema.location, Array(enclosedExpressionNode))
      }
      enclosedExpressionNode
    } else {
      expressionNode
    }
  }

  private def namedComposerExpressionToken(name: String): ComposerExpressionToken = {
    ComposerExpressionToken(ComposerExpressionTokenKind.NAME, name)
  }

  private def namedExpressionNode(kind: String, name: String, weaveLocationCapable: WeaveLocationCapable, children: Seq[ComposerExpressionNode]): ComposerExpressionNode = {
    val token = ComposerExpressionToken(kind, null)
    val nameChildNode = ComposerExpressionNode(namedComposerExpressionToken(name), weaveLocationCapable.location())
    ComposerExpressionNode(token, weaveLocationCapable.location(), (nameChildNode +: children).toArray)
  }

  def generateDWScript(expression: ComposerExpressionNode, parsingContext: ParsingContext): ComposerDWScriptResult = {
    ComposerCodeGenerator.generate(expression, parsingContext)
  }
}

case class ComposerExpression(nodes: Seq[ComposerExpressionNode])

case class ComposerValidationMessages(warnings: Seq[ValidationMessage], errors: Seq[ValidationMessage])

object ComposerExpressionParser {
  final private val supportedBinaryOpNodeIds = Set(
    AdditionOpId,
    SubtractionOpId,
    MultiplicationOpId,
    DivisionOpId,
    AsOpId)
}

object ComposerExpressionTokenKind {
  val TEXT = "TEXT"
  val PILL = "PILL"
  val TEXT_INTERPOLATION = "TEXT_INTERPOLATION"
  val NAME = "NAME"
  val UNARY_OP = "UNARY_OP"
  val BINARY_OP = "BINARY_OP"
  val FUNCTION_CALL = "FUNCTION_CALL"
  val NUMBER = "NUMBER"
  val BOOLEAN = "BOOLEAN"
  val DATE = "DATE"
  val DATE_TIME = "DATE_TIME"
  val TIME = "TIME"
  val LOCAL_TIME = "LOCAL_TIME"
  val LOCAL_DATE_TIME = "LOCAL_DATE_TIME"
  val PARENTHESIS = "PARENTHESIS"
  val ARRAY = "ARRAY"
  val TYPE = "TYPE"
  val NULL = "NULL"
}

@WeaveApi(Seq("Composer"))
case class ComposerExpressionToken(kind: String, text: String = null)

@WeaveApi(Seq("Composer"))
case class ComposerExpressionNode(token: ComposerExpressionToken, location: WeaveLocation, children: Array[ComposerExpressionNode] = Array.empty)

case class ComposerDWScriptResult(code: String, success: Boolean = true, errors: Seq[String] = Seq.empty)