package org.mule.weave.v2.formatting

import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.grammar.AdditionOpId
import org.mule.weave.v2.grammar.AsOpId
import org.mule.weave.v2.grammar.AttributeValueSelectorOpId
import org.mule.weave.v2.grammar.DivisionOpId
import org.mule.weave.v2.grammar.DynamicSelectorOpId
import org.mule.weave.v2.grammar.EqOpId
import org.mule.weave.v2.grammar.FilterSelectorOpId
import org.mule.weave.v2.grammar.GreaterThanOpId
import org.mule.weave.v2.grammar.IsOpId
import org.mule.weave.v2.grammar.LeftShiftOpId
import org.mule.weave.v2.grammar.LessOrEqualThanOpId
import org.mule.weave.v2.grammar.LessThanOpId
import org.mule.weave.v2.grammar.MetadataInjectorOpId
import org.mule.weave.v2.grammar.MultiAttributeValueSelectorOpId
import org.mule.weave.v2.grammar.MultiValueSelectorOpId
import org.mule.weave.v2.grammar.MultiplicationOpId
import org.mule.weave.v2.grammar.NotEqOpId
import org.mule.weave.v2.grammar.ObjectKeyValueSelectorOpId
import org.mule.weave.v2.grammar.RangeSelectorOpId
import org.mule.weave.v2.grammar.RightShiftOpId
import org.mule.weave.v2.grammar.SchemaValueSelectorOpId
import org.mule.weave.v2.grammar.SimilarOpId
import org.mule.weave.v2.grammar.SubtractionOpId
import org.mule.weave.v2.grammar.ValueSelectorOpId
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.annotation.SingleKeyValuePairNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.annotation.AnnotationCapableNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.conditional.IfNode
import org.mule.weave.v2.parser.ast.conditional.UnlessNode
import org.mule.weave.v2.parser.ast.functions.DoBlockNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.patterns.DeconstructArrayPatternNode
import org.mule.weave.v2.parser.ast.patterns.DeconstructObjectPatternNode
import org.mule.weave.v2.parser.ast.patterns.ExpressionPatternNode
import org.mule.weave.v2.parser.ast.patterns.LiteralPatternNode
import org.mule.weave.v2.parser.ast.patterns.PatternExpressionNode
import org.mule.weave.v2.parser.ast.patterns.PatternExpressionsNode
import org.mule.weave.v2.parser.ast.patterns.RegexPatternNode
import org.mule.weave.v2.parser.ast.patterns.TypePatternNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.AttributesNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.KeyValuePairNode
import org.mule.weave.v2.parser.ast.structure.NameValuePairNode
import org.mule.weave.v2.parser.ast.structure.ObjectNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.types.KeyTypeNode
import org.mule.weave.v2.parser.ast.types.KeyValueTypeNode
import org.mule.weave.v2.parser.ast.types.NameValueTypeNode
import org.mule.weave.v2.parser.ast.types.ObjectTypeNode
import org.mule.weave.v2.parser.ast.types.TypeParametersApplicationListNode
import org.mule.weave.v2.parser.ast.types.TypeParametersListNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionsNode
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.SimpleParserPosition

import scala.collection.mutable.ArrayBuffer

class FormattingService(options: FormattingOptions) {
  val DECONSTRUCT_SEPARATOR = " ~ "
  val TYPE_SEPARATION = ": "
  val IS_SEPARATION = " is "
  val MATCHES_SEPARATION = " matches "
  val AT_SEPARATION = " at "
  val KEY_VALUE_SEPARATION = ": "
  val EQ_SEPARATION = " = "
  val LAMBDA_SEPARATION = " -> "
  val COMMA_SEPARATION = ", "
  val SPACE_SEPARATION = " "
  val OPEN_PAREN_SPACE = "( "
  val CLOSE_PAREN_SPACE = " )"
  val OPEN_OBJECT_SPACE = "{ "
  val OPEN_CLOSE_OBJECT_SPACE = "{| "
  val OPEN_ARRAY_SPACE = "[ "
  val CLOSE_OBJECT_SPACE = " }"
  val CLOSE_CLOSE_OBJECT_SPACE = " |}"
  val CLOSE_ARRAY_SPACE = " ]"
  val CLOSE_PAREN_SEPARATION = ")"
  val OPEN_PAREN_SEPARATION = "("
  val ANNOTATION_DECLARATION = "annotation "
  val TYPE_DECLARATION = "type "
  val VAR_DECLARATION = "var "
  val FUNCTION_DECLARATION = "fun "
  val START_TYPE_REF = "<"
  val END_TYPE_REF = ">"

  val REQUIRES_WHITESPACE_OP_ID =
    Seq(
      AdditionOpId,
      IsOpId,
      SubtractionOpId,
      DivisionOpId,
      MultiplicationOpId,
      RightShiftOpId,
      LeftShiftOpId,
      EqOpId,
      NotEqOpId,
      GreaterThanOpId,
      SimilarOpId,
      LessThanOpId,
      LessOrEqualThanOpId,
      AsOpId,
      MetadataInjectorOpId)

  val SELECTION_OP_ID =
    Seq(
      AttributeValueSelectorOpId,
      MultiValueSelectorOpId,
      MultiAttributeValueSelectorOpId,
      DynamicSelectorOpId,
      SchemaValueSelectorOpId,
      ValueSelectorOpId,
      ObjectKeyValueSelectorOpId,
      FilterSelectorOpId,
      RangeSelectorOpId)

  def format(ast: AstNode, document: WeaveTextDocument): Unit = {
    format(ast, document, 0, -1)
  }

  def calculateBinaryFunctionOffset(firstArg: AstNode): Int = {
    firstArg match {
      case fcn: FunctionCallNode if (isInfixBinaryCall(fcn)) => {
        calculateBinaryFunctionOffset(fcn.args.args.head)
      }
      case n => resolveLocation(n).endPosition.column
    }
  }

  private def format(ast: AstNode, document: WeaveTextDocument, offset: Int, previousLine: Int): Unit = {

    if (AstNodeHelper.isInjectedNode(ast)) {
      format(ast.children(), document, offset, previousLine)
      return
    }

    if (ast.isAnnotatedWith(classOf[EnclosedMarkAnnotation])) {
      val enclosedMarkAnnotations: Seq[EnclosedMarkAnnotation] = ast.annotationsBy(classOf[EnclosedMarkAnnotation]).sortBy(_.location.startPosition.index)
      var newStartPreviousLine = previousLine
      var newStartOffset = offset
      val offsets = new ArrayBuffer[Int]()
      enclosedMarkAnnotations.foreach((ea) => {
        val startLine = ea.location.startPosition.line
        offsets += (newStartOffset)
        if (newStartPreviousLine != startLine) {
          applyIndent(ea.location.startPosition, document, newStartOffset)
          newStartPreviousLine = startLine
          newStartOffset = indentOffset(newStartOffset)
        }
      })
      doFormat(ast, document, offset = newStartOffset, previousLine = newStartPreviousLine)
      //
      var newEndPreviousLine = ast.location().endPosition.line
      enclosedMarkAnnotations.reverse.zipWithIndex.foreach((ea) => {
        val parenNode = ea._1
        if (parenNode.location.endPosition.line != newEndPreviousLine) {
          val parenOffset = offsets(offsets.length - (ea._2 + 1))
          applyIndent(parenNode.location.endPosition, document, parenOffset)
          newEndPreviousLine = parenNode.location.endPosition.line
        }
      })
    } else {
      doFormat(ast, document, offset, previousLine)
    }
  }

  private def doFormat(ast: AstNode, document: WeaveTextDocument, offset: Int, previousLine: Int): Unit = {
    val newPreviousLine =
      ast match {
        case an: AnnotationCapableNode if an.codeAnnotations.nonEmpty => {
          val annotations = an.codeAnnotations
          annotations
            .filterNot(AstNodeHelper.isInjectedNode)
            .sortBy(_.location().startPosition.index)
            .foldLeft(previousLine)((line, child) => {
              format(child, document, offset, line)
              child.location().endPosition.line
            })
        }
        case _ => previousLine
      }

    val startLine: Int = ast.location().startPosition.line //In here we don't use the paren
    if (startLine > newPreviousLine) {
      applyIndent(ast.location().startPosition, document, offset)
    }

    ast match {
      case _: DocumentNode | _: ModuleNode => {
        format(getChildrenWithoutAnnotation(ast), document, offset, 0)
      }
      case fn: FunctionNode => {
        formatFunction(fn, document, offset, startLine, functionSeparation = LAMBDA_SEPARATION)
      }
      case fd: FunctionDirectiveNode => {
        if (fd.codeAnnotations.isEmpty) {
          separateBy(fd.location().startPosition, fd.variable.location().startPosition, FUNCTION_DECLARATION, document)
        }
        fd.literal match {
          case fn: FunctionNode => {
            formatFunction(fn, document, offset, fd.variable.location().endPosition.line, functionSeparation = EQ_SEPARATION)
          }
          case node => format(node, document, offset, fd.variable.location().endPosition.line)
        }
      }
      case fp: FunctionParameter => {
        if (fp.wtype.isDefined) {
          separateBy(fp.variable, fp.wtype.get, TYPE_SEPARATION, document)
          format(fp.wtype.get, document, offset, fp.wtype.get.location().startPosition.line)
        }
        if (fp.defaultValue.isDefined) {
          val leftPosition = fp.wtype
            .map(_.location().endPosition)
            .getOrElse(fp.variable.location().endPosition)
          val defaultParamValue = fp.defaultValue.get
          separateBy(leftPosition, resolveLocation(defaultParamValue).startPosition, EQ_SEPARATION, document)
          format(defaultParamValue, document, offset, defaultParamValue.location().startPosition.line)
        }
      }

      case vd: VarDirective => {
        if (vd.codeAnnotations.isEmpty) {
          separateBy(vd.location().startPosition, vd.variable.location().startPosition, VAR_DECLARATION, document)
        }
        if (vd.wtype.isDefined) {
          separateBy(vd.variable, vd.wtype.get, TYPE_SEPARATION, document)
        }
        separateBy(vd.variable, vd.value, EQ_SEPARATION, document)
        format(vd.value, document, indentOffset(offset), vd.variable.location().startPosition.line)
      }
      case vd: TypeDirective => {
        if (vd.codeAnnotations.isEmpty) {
          separateBy(vd.location().startPosition, vd.variable.location().startPosition, TYPE_DECLARATION, document)
        }
        val typeParametersListNode: Option[TypeParametersListNode] = vd.typeParametersListNode
        if (typeParametersListNode.isDefined && typeParametersListNode.get.typeParameters.nonEmpty) {
          val typeParamList: TypeParametersListNode = typeParametersListNode.get
          separateBy(vd.variable.location().endPosition, typeParamList.typeParameters.head.location().startPosition, START_TYPE_REF, document)
          separateBy(typeParamList.typeParameters, COMMA_SEPARATION, document)
          separateBy(typeParamList.typeParameters.last.location().endPosition, vd.typeExpression.location().startPosition, END_TYPE_REF + " " + EQ_SEPARATION, document)
        } else {
          separateBy(vd.variable, vd.typeExpression, EQ_SEPARATION, document)
        }
        format(vd.typeExpression, document, offset, vd.variable.location().endPosition.line)
      }
      case schemaNode: SchemaNode => {
        if (schemaNode.properties.nonEmpty) {
          val firstChild = schemaNode.properties.head
          val lastChild = schemaNode.properties.last
          if (schemaNode.location().startPosition.line == resolveLocation(firstChild).startPosition.line) {
            separateBy(schemaNode.location().startPosition, resolveLocation(firstChild).startPosition, OPEN_OBJECT_SPACE, document)
          }
          separateBy(schemaNode.properties, COMMA_SEPARATION, document)
          format(schemaNode.properties, document, indentOffset(offset), startLine)

          if (schemaNode.location().endPosition.line == resolveLocation(lastChild).startPosition.line) {
            separateBy(resolveLocation(lastChild).endPosition, schemaNode.location().endPosition, CLOSE_OBJECT_SPACE, document)
          }

          indentClose(document, offset, schemaNode)
        }
      }
      case vd: AnnotationDirectiveNode => {
        separateBy(vd.location().startPosition, vd.nameIdentifier.location().startPosition, ANNOTATION_DECLARATION, document)
      }
      //Types
      case otn: ObjectTypeNode => {
        if (otn.properties.nonEmpty) {
          val firstChild = otn.properties.head
          val lastChild = otn.properties.last
          if (otn.location().startPosition.line == resolveLocation(firstChild).startPosition.line) {
            separateBy(otn.location().startPosition, resolveLocation(firstChild).startPosition, if (otn.close) OPEN_CLOSE_OBJECT_SPACE else OPEN_OBJECT_SPACE, document)
          }
          separateBy(otn.properties, COMMA_SEPARATION, document)
          format(otn.properties, document, indentOffset(offset), startLine)

          if (otn.location().endPosition.line == resolveLocation(lastChild).startPosition.line) {
            separateBy(resolveLocation(lastChild).endPosition, otn.location().endPosition, if (otn.close) CLOSE_CLOSE_OBJECT_SPACE else CLOSE_OBJECT_SPACE, document)
          }

          indentClose(document, offset, otn)
        }
      }
      case kvp: NameValueTypeNode => {
        separateBy(kvp.name, kvp.value, KEY_VALUE_SEPARATION, document)
        format(getChildrenWithoutAnnotation(kvp), document, indentOffset(offset), startLine)
      }
      case kvp: KeyValueTypeNode => {
        separateBy(kvp.key, kvp.value, KEY_VALUE_SEPARATION, document)
        format(getChildrenWithoutAnnotation(kvp), document, indentOffset(offset), startLine)
      }
      case kvp: KeyTypeNode => {
        separateBy(kvp.attrs, COMMA_SEPARATION, document)
        format(getChildrenWithoutAnnotation(kvp), document, offset, startLine)
      }
      case rt: TypeReferenceNode => {
        if (rt.typeArguments.isDefined && rt.typeArguments.get.nonEmpty) {
          if (!AstNodeHelper.isInjectedNode(rt.typeArguments.get.head)) {
            separateBy(rt.typeArguments.get, COMMA_SEPARATION, document)
            separateBy(rt.variable.location().endPosition, rt.typeArguments.get.head.location().startPosition, START_TYPE_REF, document)
            separateBy(rt.typeArguments.get.last.location().endPosition, rt.location().endPosition, END_TYPE_REF, document)
          }
        }
        format(getChildrenWithoutAnnotation(rt), document, indentOffset(offset), startLine)
      }
      case fcn: FunctionCallNode => {
        formatFunctionCall(fcn, document, offset, startLine)
      }
      case tpal: TypeParametersApplicationListNode => {
        separateBy(tpal.location().startPosition, tpal.typeParameters.head.location().startPosition, START_TYPE_REF, document)
        separateBy(tpal.typeParameters, COMMA_SEPARATION, document)
        format(tpal.typeParameters, document, offset, previousLine)
        separateBy(tpal.typeParameters.last.location().endPosition, tpal.location().endPosition, END_TYPE_REF, document)

      }
      case fcp: FunctionCallParametersNode => {
        if (fcp.args.isEmpty) {
          separateBy(fcp.location().startPosition, fcp.location().endPosition, OPEN_PAREN_SEPARATION + CLOSE_PAREN_SEPARATION, document)
        } else {
          separateBy(fcp.location().startPosition, fcp.args.head.location().startPosition, OPEN_PAREN_SEPARATION, document)
          separateBy(fcp.args, COMMA_SEPARATION, document)
          format(fcp.args, document, offset, previousLine)
          separateBy(fcp.args.last.location().endPosition, fcp.location().endPosition, CLOSE_PAREN_SEPARATION, document)
        }
      }
      case on: ObjectNode if (on.elements.isEmpty) =>
        indentClose(document, offset, on)
      //Object Values
      case on: ObjectNode => {
        val firstChild = on.elements.head
        val lastChild = on.elements.last
        if (!on.isAnnotatedWith(classOf[SingleKeyValuePairNodeAnnotation]) && on.location().startPosition.line == resolveLocation(firstChild).startPosition.line) {
          separateBy(on.location().startPosition, resolveLocation(firstChild).startPosition, OPEN_OBJECT_SPACE, document)
        }
        separateBy(on.elements, COMMA_SEPARATION, document)
        format(on.elements, document, indentOffset(offset), startLine)

        if (!on.isAnnotatedWith(classOf[SingleKeyValuePairNodeAnnotation]) && on.location().endPosition.line == resolveLocation(lastChild).startPosition.line) {
          separateBy(resolveLocation(lastChild).endPosition, on.location().endPosition, CLOSE_OBJECT_SPACE, document)
        }
        indentClose(document, offset, on)
      }
      case kvp: KeyValuePairNode => {
        separateBy(kvp.key, kvp.value, KEY_VALUE_SEPARATION, document)
        if (kvp.cond.isDefined) {
          separateBy(kvp.value, kvp.cond.get, ") if ", document)
        }
        format(getChildrenWithoutAnnotation(kvp), document, indentOffset(offset), startLine)
      }
      case an: AttributesNode => {
        val paramList = an.children()
        separateBy(paramList, COMMA_SEPARATION, document)
        format(getChildrenWithoutAnnotation(an), document, offset, startLine)
        indentClose(document, offset, an)
      }
      case kvp: NameValuePairNode => {
        separateBy(kvp.key, kvp.value, KEY_VALUE_SEPARATION, document)
        format(getChildrenWithoutAnnotation(kvp), document, indentOffset(offset), startLine)
      }
      //Flow Controls
      case doBlockNode: DoBlockNode => {
        val headerNode = doBlockNode.header
        val bodyNode = doBlockNode.body
        format(headerNode, document, indentOffset(offset), startLine)
        val bodyStartPosition = resolveLocation(bodyNode).startPosition
        val headerEndPosition = headerNode.location().endPosition
        if (!AstNodeHelper.isInjectedNode(headerNode)) {
          if (headerEndPosition.line != bodyStartPosition.line) {
            applyIndent(headerEndPosition, document, indentOffset(offset))
          }
        }
        format(bodyNode, document, indentOffset(offset), startLine)
        indentClose(document, offset, doBlockNode)
      }
      case binaryOpNode: BinaryOpNode if (REQUIRES_WHITESPACE_OP_ID.contains(binaryOpNode.binaryOpId)) => {
        separateBy(binaryOpNode.lhs, binaryOpNode.rhs, " " + binaryOpNode.binaryOpId.name + " ", document)
        format(getChildrenWithoutAnnotation(binaryOpNode), document, offset, startLine)
      }
      case binaryOpNode: BinaryOpNode if (SELECTION_OP_ID.contains(binaryOpNode.binaryOpId)) => {
        val lhs = binaryOpNode.lhs
        val rhs = binaryOpNode.rhs
        forceNoSpaceBetween(lhs, rhs, document)
        format(lhs, document, offset, startLine)
        format(rhs, document, indentOffset(offset), startLine)
      }
      case on: IfNode => {
        format(on.condition, document, indentOffset(offset), startLine)
        format(on.ifExpr, document, indentOffset(offset), resolveLocation(on.condition).endPosition.line)
        if (!on.elseExpr.isInstanceOf[IfNode]) {
          //Line Number where the else keyword is
          var elseLineNumber = resolveLocation(on.ifExpr).endPosition.line
          //Search for the else keyword
          val textBetween: String = document.text(resolveLocation(on.ifExpr).endPosition.index, resolveLocation(on.elseExpr).startPosition.index)
          val lines = textBetween.linesIterator
          var location = resolveLocation(on.ifExpr).endPosition.index //end line
          var elseFound = false
          while (lines.hasNext && !elseFound) {
            val line = lines.next()
            if (line.trim.startsWith("else")) {
              elseFound = true
              if (elseLineNumber != resolveLocation(on.elseExpr).startPosition.line) {
                applyIndent(document, line, location, offset)
              }
            } else {
              elseLineNumber = elseLineNumber + 1
              location = location + line.length + 1
            }
          }
          if (elseLineNumber != resolveLocation(on.elseExpr).startPosition.line) {
            format(on.elseExpr, document, indentOffset(offset), resolveLocation(on.ifExpr).endPosition.line)
          } else {
            format(on.elseExpr, document, offset, resolveLocation(on.ifExpr).endPosition.line)
          }
        } else {
          format(on.elseExpr, document, offset, resolveLocation(on.ifExpr).endPosition.line)
        }
      }
      case on: UnlessNode => {
        format(getChildrenWithoutAnnotation(on), document, indentOffset(offset), startLine)
      }
      case an: ArrayNode if (an.elements.isEmpty) => {
        indentClose(document, offset, an)
      }
      case an: ArrayNode => {
        val lastChild = an.elements.last
        val firstChild = an.elements.head
        if (an.location().startPosition.line == resolveLocation(firstChild).startPosition.line) {
          separateBy(an.location().startPosition, resolveLocation(firstChild).startPosition, OPEN_ARRAY_SPACE, document)
        }
        separateBy(an.elements, COMMA_SEPARATION, document)
        format(an.elements, document, indentOffset(offset), startLine)
        indentClose(document, offset, an)
        if (an.location().endPosition.line == resolveLocation(lastChild).startPosition.line) {
          separateBy(resolveLocation(lastChild).endPosition, an.location().endPosition, CLOSE_ARRAY_SPACE, document)
        }
      }
      case patternExpressionsNode: PatternExpressionsNode => {
        format(getChildrenWithoutAnnotation(patternExpressionsNode), document, indentOffset(offset), startLine)
        indentClose(document, offset, patternExpressionsNode)
      }
      case patternExpressionsNode: PatternExpressionNode => {
        patternExpressionsNode match {
          case RegexPatternNode(pattern, name, onMatch) => {
            if (!AstNodeHelper.isInjectedNode(name)) {
              separateBy(name, pattern, MATCHES_SEPARATION, document)
            }
            format(pattern, document, offset, patternExpressionsNode.location().startPosition.line)
          }
          case TypePatternNode(pattern, name, _) => {
            if (!AstNodeHelper.isInjectedNode(name)) {
              separateBy(name, pattern, IS_SEPARATION, document)
            }
            format(pattern, document, offset, patternExpressionsNode.location().startPosition.line)
          }
          case LiteralPatternNode(pattern, name, _) => {
            if (!AstNodeHelper.isInjectedNode(name)) {
              separateBy(name, pattern, TYPE_SEPARATION, document)
            }
            format(pattern, document, offset, patternExpressionsNode.location().startPosition.line)
          }
          case ExpressionPatternNode(pattern, _, _) => {
            format(pattern, document, offset, patternExpressionsNode.location().startPosition.line)
          }
          case DeconstructArrayPatternNode(head, tail, _) => {
            separateBy(head, tail, DECONSTRUCT_SEPARATOR, document)
          }
          case DeconstructObjectPatternNode(headKey, headValue, tail, _) => {
            separateBy(headKey, headValue, KEY_VALUE_SEPARATION, document)
            separateBy(headValue, tail, DECONSTRUCT_SEPARATOR, document)
          }
          case _ => {}
        }
        format(patternExpressionsNode.onMatch, document, indentOffset(offset), startLine)
      }
      case updateExpressionsNode: UpdateExpressionsNode => {
        format(getChildrenWithoutAnnotation(updateExpressionsNode), document, indentOffset(offset), startLine)
        indentClose(document, offset, updateExpressionsNode)
      }
      case updateExpressionNode: UpdateExpressionNode => {
        if (!AstNodeHelper.isInjectedNode(updateExpressionNode.name)) {
          separateBy(updateExpressionNode.name, updateExpressionNode.selector, AT_SEPARATION, document)
        }
        if (updateExpressionNode.condition.isDefined) {
          format(updateExpressionNode.condition.get, document, offset, updateExpressionNode.condition.get.location().startPosition.line)
        }
        format(updateExpressionNode.updateExpression, document, indentOffset(offset), startLine)
      }
      case node => {
        format(getChildrenWithoutAnnotation(node), document, offset, startLine)
      }
    }
  }

  private def getChildrenWithoutAnnotation(astNode: AstNode) = {
    astNode.children().filter(node => !node.isInstanceOf[AnnotationNode])
  }

  private def isInfixBinaryCall(fcn: FunctionCallNode) = {
    AstNodeHelper.isInfixFunctionCall(fcn) && fcn.args.args.length == 2
  }

  private def isCustomInterpolation(fcn: FunctionCallNode) = {
    AstNodeHelper.isCustomInterpolatedNode(fcn)
  }

  /**
    * Increase the offset by the amount of spaces or tabs required
    *
    * @return The new indented offset
    */
  private def indentOffset(offset: Int) = {
    //If we are using tabs then it should be one
    offset + (if (options.insertSpaces) options.tabSize else 1)
  }

  def separateBy(nodes: Seq[AstNode], separator: String, document: WeaveTextDocument): Unit = {
    var i = 1
    while (i < nodes.size) {
      separateBy(nodes(i - 1), nodes(i), separator, document)
      i = i + 1
    }
  }

  private def indentClose(document: WeaveTextDocument, indent: Int, astNode: AstNode): Unit = {
    val startPosition: Position = astNode
      .children()
      .lastOption
      .map(resolveLocation(_).endPosition)
      .getOrElse(resolveLocation(astNode).startPosition)
    val endPosition: Position = resolveLocation(astNode).endPosition
    if (startPosition.line != endPosition.line) {
      val leftPosition = SimpleParserPosition(endPosition.index - 1, endPosition.line, endPosition.column - 1, endPosition.source)
      applyIndent(leftPosition, document, indent)
    }
  }

  private def format(nodes: Seq[AstNode], document: WeaveTextDocument, indent: Int, startLine: Int): Unit = {
    nodes
      .filterNot(AstNodeHelper.isInjectedNode)
      .sortBy(resolveLocation(_).startPosition.index)
      .foldLeft(startLine)((line, child) => {
        format(child, document, indent, line)
        resolveLocation(child).endPosition.line
      })
  }

  private def formatFunctionCall(fcn: FunctionCallNode, document: WeaveTextDocument, offset: Int, line: Int): Unit = {
    if (isInfixBinaryCall(fcn)) {
      val firstArg = fcn.args.args.head
      val secondArg = fcn.args.args.last
      format(firstArg, document, offset, line)
      forceOneSpaceBetween(firstArg, fcn.function, document)

      val functionOffset: Int = indentOffset(offset)
      val secondParamOffset: Int = indentOffset(functionOffset)

      format(fcn.function, document, functionOffset, resolveLocation(firstArg).endPosition.line)

      if (fcn.typeParameters.isDefined && fcn.typeParameters.nonEmpty) {
        val typeParametersList = fcn.typeParameters.get
        forceNoSpaceBetween(fcn.function, typeParametersList, document)
        format(typeParametersList, document, functionOffset, resolveLocation(fcn.function).endPosition.line)
        forceOneSpaceBetween(typeParametersList, secondArg, document)
        format(secondArg, document, secondParamOffset, resolveLocation(typeParametersList).endPosition.line)
      } else {
        forceOneSpaceBetween(fcn.function, secondArg, document)
        format(secondArg, document, secondParamOffset, resolveLocation(fcn.function).endPosition.line)
      }
    } else {
      val params = fcn.args
      format(fcn.function, document, offset, line)

      if (fcn.typeParameters.isDefined && fcn.typeParameters.nonEmpty) {
        val typeParametersList = fcn.typeParameters.get
        forceNoSpaceBetween(fcn.function, typeParametersList, document)
        format(typeParametersList, document, offset, resolveLocation(fcn.function).endPosition.line)
        forceNoSpaceBetween(typeParametersList, params, document)
      } else if (!isCustomInterpolation(fcn)) {
        forceNoSpaceBetween(fcn.function, params, document)
      }
      format(params, document, indentOffset(offset), resolveLocation(fcn.function).endPosition.line)
    }
  }

  private def formatFunction(fn: FunctionNode, document: WeaveTextDocument, offset: Int, line: Int, functionSeparation: String): Unit = {
    val paramList: Seq[FunctionParameter] = fn.params.paramList
    separateBy(paramList, COMMA_SEPARATION, document)

    //Format Parameters
    val paramsOffset = fn.params.location().startPosition.column
    format(paramList, document, paramsOffset, fn.params.location().startPosition.line)

    if (fn.returnType.isDefined) {
      separateBy(fn.params, fn.returnType.get, TYPE_SEPARATION, document)
      separateBy(fn.returnType.get, fn.body, functionSeparation, document)
    } else {
      separateBy(fn.params, fn.body, functionSeparation, document)
    }

    val headParam: Option[FunctionParameter] = fn.params.paramList.headOption
    if (headParam.isDefined) {
      separateBy(fn.params.location().startPosition, headParam.get.location().startPosition, OPEN_PAREN_SEPARATION, document)
    }

    val lastParameter: Option[FunctionParameter] = fn.params.paramList.lastOption
    if (lastParameter.isDefined) {
      separateBy(lastParameter.get.location().endPosition, fn.params.location().endPosition, CLOSE_PAREN_SEPARATION, document)
    }

    format(fn.body, document, indentOffset(offset), line)
  }

  private def forceOneSpaceBetween(leftNode: AstNode, rightNode: AstNode, document: WeaveTextDocument): Unit = {
    val leftLocation = resolveLocation(leftNode)
    val rightLocation = resolveLocation(rightNode)
    if (leftLocation.endPosition.line == rightLocation.startPosition.line) {
      val nextChar: String = document.text(leftLocation.endPosition.index, rightLocation.startPosition.index)
      val i = countInitialWhiteSpaces(nextChar)
      if (i > 1) {
        document.delete(leftLocation.endPosition.index, leftLocation.endPosition.index + (i - 1))
      } else if (i == 0) {
        document.insert(" ", leftLocation.endPosition.index)
      }
    }
  }

  /**
    * Resolves the location of the given node taken into account the parenthesis
    *
    * @param astNode The node to look for
    * @return The location
    */
  private def resolveLocation(astNode: AstNode) = {
    if (astNode.isAnnotatedWith(classOf[EnclosedMarkAnnotation])) {
      astNode.annotationsBy(classOf[EnclosedMarkAnnotation]).last.location
    } else {
      astNode.location()
    }
  }

  private def forceNoSpaceBetween(leftNode: AstNode, rightNode: AstNode, document: WeaveTextDocument): Unit = {
    if (!AstNodeHelper.isInjectedNode(leftNode) && !AstNodeHelper.isInjectedNode(rightNode)) {
      val leftLocation = resolveLocation(leftNode)
      val rightLocation = resolveLocation(rightNode)
      if (leftLocation.endPosition.line == rightLocation.startPosition.line) {
        val nextChar: String = document.text(leftLocation.endPosition.index, rightLocation.startPosition.index)
        val i = countInitialWhiteSpaces(nextChar)
        if (i > 0) {
          document.delete(leftLocation.endPosition.index, leftLocation.endPosition.index + (i))
        }
      }
    }
  }

  private def separateBy(leftNode: AstNode, rightNode: AstNode, expectedSeparation: String, document: WeaveTextDocument): Unit = {
    if (!AstNodeHelper.isInjectedNode(leftNode) && !AstNodeHelper.isInjectedNode(rightNode)) {
      val leftLocation = resolveLocation(leftNode)
      val rightLocation = resolveLocation(rightNode)
      separateBy(leftLocation.endPosition, rightLocation.startPosition, expectedSeparation, document)
    }
  }

  private def separateBy(leftLocation: Position, rightLocation: Position, expectedSeparation: String, document: WeaveTextDocument): Unit = {
    if (leftLocation.line == rightLocation.line) {
      val startIndex: Int = leftLocation.index
      val endIndex: Int = rightLocation.index
      val separation: String = document.text(startIndex, endIndex)
      //If the current separation and the expected one not just diff by whitespaces then we shouldn't do anything
      //For example there may be comments and we should then keep it as it is?
      if (equalsIgnoreWhiteSpaces(expectedSeparation, separation)) {
        if (!separation.equals(expectedSeparation)) {
          if (leftLocation.line != rightLocation.line) {
            val rightTrim = separation.replaceAll("\\s+$", "")
            document.replace(startIndex, startIndex + rightTrim.length, expectedSeparation)
          } else {
            document.replace(startIndex, endIndex, expectedSeparation)
          }
        }
      } else {
        println(separation.trim + " != " + expectedSeparation)
      }
    }
  }

  private def equalsIgnoreWhiteSpaces(expectedSeparation: String, separation: String) = {
    separation.replaceAll("\\s+", "").equals(expectedSeparation.replaceAll("\\s+", ""))
  }

  private def applyIndent(ast: AstNode, document: WeaveTextDocument, column: Int): Unit = {
    val startPosition: Position = resolveLocation(ast).startPosition
    applyIndent(startPosition, document, column)
  }

  def countInitialWhiteSpaces(text: String): Int = {
    var i = 0
    while (i < text.length && text.charAt(i).isWhitespace) {
      i = i + 1
    }
    i
  }

  private def applyIndent(startPosition: Position, document: WeaveTextDocument, expectedIndent: Int): Unit = {
    val actualColumn = startPosition.column - 1 //remove 1 as column is one based
    val startLineIndex = startPosition.index - actualColumn
    val text = document.text(startLineIndex, startPosition.index)
    applyIndent(document, text, startLineIndex, expectedIndent)
  }

  private def applyIndent(document: WeaveTextDocument, text: String, startLineIndex: Int, expectedIndent: Int) = {
    val currentIndent = countInitialWhiteSpaces(text)
    if (currentIndent != expectedIndent) {
      val diffSpaces: Int = expectedIndent - currentIndent
      if (diffSpaces < 0) {
        document.delete(startLineIndex, startLineIndex - diffSpaces)
      } else {
        val spaces = if (options.insertSpaces) " " else "\t"
        document.insert(spaces * diffSpaces, startLineIndex)
      }
    }
  }
}

case class FormattingOptions(tabSize: Int = 2, insertSpaces: Boolean = false)
