package org.mule.weave.v2.inspector

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.grammar.AsOpId
import org.mule.weave.v2.grammar.EqOpId
import org.mule.weave.v2.grammar.SimilarOpId
import org.mule.weave.v2.inspector.ReferenceUtils.isReferencingTo
import org.mule.weave.v2.parser.UsingTypeOfToCompareTypes
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.NameIdentifier.CORE_MODULE
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware
import org.mule.weave.v2.scope.ScopesNavigator

object TypeOfInspector extends CodeInspector[AstNodeResultAware[_] with ScopeNavigatorResultAware] {

  override def inspect(node: AstNode, scopeData: AstNodeResultAware[_] with ScopeNavigatorResultAware, parsingContext: ParsingContext): Unit = {
    node match {
      case BinaryOpNode(op, lhs, StringNode(typeName, _), _) => {
        lhs match {
          case FunctionCallNode(VariableReferenceNode(variable, _), args, _, _) if (isReferencingTypeOf(variable, scopeData.scope) && op == SimilarOpId) => {
            //typeOf(expr) ~= "Object"
            if (args.args.length == 1) {
              parsingContext.messageCollector.warning(UsingTypeOfToCompareTypes(args.args.head, typeName, node), node.location())
            }
          }
          case BinaryOpNode(AsOpId, FunctionCallNode(VariableReferenceNode(variable, _), args, _, _), TypeReferenceNode(NameIdentifier("String", _), _, _, _, _), _) if (isReferencingTypeOf(variable, scopeData.scope) && (op == EqOpId || op == SimilarOpId)) => {
            //typeOf(expr) as String == "Object"
            if (args.args.length == 1) {
              parsingContext.messageCollector.warning(UsingTypeOfToCompareTypes(args.args.head, typeName, node), node.location())
            }
          }
          case _ =>
        }
      }
      case BinaryOpNode(op, lhs, VariableReferenceNode(typeName, _), _) => {
        lhs match {
          case FunctionCallNode(VariableReferenceNode(variable, _), args, _, _) if (isReferencingTypeOf(variable, scopeData.scope) && op == EqOpId) => {
            //typeOf(expr) == Object
            if (args.args.length == 1) {
              parsingContext.messageCollector.warning(UsingTypeOfToCompareTypes(args.args.head, typeName.name, node), node.location())
            }
          }
          case _ =>
        }
      }
      case _ =>
    }
  }

  private def isReferencingTypeOf(variable: NameIdentifier, scopeData: ScopesNavigator) = {
    isReferencingTo(variable, CORE_MODULE.::("typeOf"), scopeData)
  }
}

class TypeOfQuickFixAction(expression: AstNode, typeName: String, containerNode: AstNode) extends QuickFixAction {
  override def run(document: WeaveTextDocument): Unit = {
    val locationToUpdate = containerNode.location()
    val startPosition = locationToUpdate.startPosition.index
    document.delete(startPosition, locationToUpdate.endPosition.index)
    document.insert(CodeGenerator.generate(expression) + s" is ${typeName}", startPosition)
  }
}
