package org.mule.weave.v2.mapping

import org.mule.weave.v2.mapping.QNameType.QNameType
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.ParsingContextProvider
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.ts.WeaveType

import java.util.Comparator
import scala.annotation.tailrec
import scala.collection.mutable

/**
  * This builder keeps track of the assignments and when build() is called it creates a Mapping with them.
  */
class DataMappingEditor(parsingContextProvider: ParsingContextProvider) {
  //TODO: this should have the source and target metadata
  //TODO: add a HistoryManager to keep track of the changes
  //TODO: maybe memoize previously created Mappings

  private val exprAssignments = mutable.ListBuffer[ExpressionAssignment]()

  private val arrowAssignments = new java.util.TreeSet[(Int, FieldAssignment)](new ArrowComparator())

  private var arrowCount: Int = 0

  def build(): DataMapping = {
    val rootMapping = createRootMapping()

    val it = arrowAssignments.iterator()
    while (it.hasNext) {
      val next = it.next()
      applyArrowAssignment(next._2, rootMapping)
    }

    for (assignment <- exprAssignments) {
      applyExprAssignment(assignment, rootMapping)
    }
    rootMapping
  }

  def loadFrom(script: String): DataMappingEditor = {
    val loader = new DataMappingLoader(parsingContextProvider.parsingContext(NameIdentifier.anonymous))
    val assignments: Seq[FieldAssignment] = loader.inferAssignments(WeaveResource.anonymous(script))
    assignments.foreach((as) => {
      addAssignment(as)
    })
    this
  }

  def addAssignment(source: String, target: String): Unit = {
    val sourceQName = NamePathElement.fromString(source)
    val targetQName = NamePathElement.fromString(target)
    if (canAssign(sourceQName, targetQName)) {
      val arrow = new FieldAssignment(sourceQName, targetQName)
      addAssignment(arrow)
    } else {
      throw new RuntimeException(s"Invalid assignment: $source -> $target")
    }
  }

  def addAssignment(source: String, target: String, sourceTypeStr: String, targetTypeStr: String): Unit = {
    val sourceQName: NamePathElement = NamePathElement.fromString(source)
    val targetQName: NamePathElement = NamePathElement.fromString(target)
    if (canAssign(sourceQName, targetQName)) {
      val sourceType: Option[WeaveType] = WeaveType.getSimpleType(sourceTypeStr)
      val targetType: Option[WeaveType] = WeaveType.getSimpleType(targetTypeStr)
      val arrow = new FieldAssignment(sourceQName, targetQName, sourceType, targetType)
      addAssignment(arrow)
    } else {
      throw new RuntimeException(s"Invalid assignment: $source -> $target")
    }
  }

  def addExpression(target: String, expr: String): Unit = {
    val targetQName = NamePathElement.fromString(target)
    val phaseResult = parseMapping(expr)
    if (phaseResult.hasErrors()) {
      throw new RuntimeException(s"Invalid expression")
    } else {
      val exprAstNode = phaseResult.getResult().astNode.root
      exprAssignments += ExpressionAssignment(targetQName, exprAstNode)
    }
  }

  def addExpression(target: String, expr: String, sourceTypeStr: String, targetTypeStr: String): Unit = {
    val targetQName = NamePathElement.fromString(target)
    val phaseResult = parseMapping(expr)
    if (phaseResult.hasErrors()) {
      throw new RuntimeException(s"Invalid expression")
    } else {
      val sourceType = WeaveType.getSimpleType(sourceTypeStr)
      val targetType = WeaveType.getSimpleType(targetTypeStr)
      val exprAstNode = phaseResult.getResult().astNode.root
      exprAssignments += ExpressionAssignment(targetQName, exprAstNode, sourceType, targetType)
    }
  }

  def addAssignment(assignment: FieldAssignment): Unit = {
    arrowAssignments.add((arrowCount, assignment))
    arrowCount += 1
  }

  def remove(source: String, target: String): Boolean = {
    val assignment = FieldAssignment(source, target)
    remove(assignment)
  }

  def remove(assignment: FieldAssignment): Boolean = {
    arrowAssignments.remove(assignment)
  }

  def clear(): Unit = {
    arrowAssignments.clear()
  }

  protected def parseMapping(content: String): PhaseResult[ParsingResult[DocumentNode]] = {
    MappingParser.parse(MappingParser.parsingPhase(), WeaveResource.anonymous(content), parsingContextProvider.parsingContext(NameIdentifier.anonymous))
  }

  private def createRootMapping(): DataMapping = {
    if (arrowAssignments.isEmpty) {
      DataMapping(None, new RootNamePathElement(QNameType.Object), new RootNamePathElement(QNameType.Object))
    } else {
      val sourceRoot = arrowAssignments.first._2.source.path().head
      val targetRoot = arrowAssignments.first._2.target.path().head
      DataMapping(None, sourceRoot, targetRoot)
    }
  }

  /**
    * This method adds the expression mappings to the rootMapping
    */
  private def applyExprAssignment(assignment: ExpressionAssignment, mapping: DataMapping): Unit = {
    val target = assignment.target
    //look for the most specific target that "contains" the target
    val moreSpecific = mapping.childMappings().filter(x => x.target.contains(target))
    if (moreSpecific.isEmpty) {
      mapping.addExpression(target, assignment.expressionNode, assignment.sourceType, assignment.targetType)
    } else {
      moreSpecific.foreach(x => applyExprAssignment(assignment, x))
    }
  }

  /**
    * This method mutates the rootMapping according to the given assignment.
    */
  private def applyArrowAssignment(assignment: FieldAssignment, rootMapping: DataMapping): Unit = {
    val source = assignment.source
    val target = assignment.target
    val sourcePath = source.path()
    val targetPath = target.path()
    if (canAssign(source, target)) {
      val cardinalityDiff = target.cardinality - source.cardinality
      if (cardinalityDiff > 0) {
        val (startingTarget: Array[NamePathElement], mapping: DataMapping) = getStartingTarget(cardinalityDiff, targetPath, rootMapping)
        applyArrowAssignment(sourcePath, startingTarget, mapping, assignment.sourceType, assignment.targetType)
      } else {
        applyArrowAssignment(sourcePath, targetPath, rootMapping, assignment.sourceType, assignment.targetType)
      }
    } else {
      throw new RuntimeException(s"Invalid assignment: $source -> $target")
    }
  }

  @tailrec
  private def getStartingTarget(cardinalityDiff: Int, targetPath: Array[NamePathElement], mapping: DataMapping): (Array[NamePathElement], DataMapping) = {
    if (cardinalityDiff == 0) {
      return (targetPath, mapping)
    }
    val nextRepeated = fromNextRepeated(targetPath)
    val innerMapping = mapping.childMappings().find(x => x.target == nextRepeated.head)
      .getOrElse(mapping)

    getStartingTarget(cardinalityDiff - 1, nextRepeated.tail, innerMapping)
  }

  @tailrec
  private def applyArrowAssignment(sourcePath: Array[NamePathElement], targetPath: Array[NamePathElement], mapping: DataMapping, sourceType: Option[WeaveType], targetType: Option[WeaveType]): Unit = {
    // advance through the sourcePath until a repetitive element is found
    // create intermediate mappings that don't exist when repetitive elements are found (in both sides)
    // add arrows in the corresponding mapping

    //TODO: can't disambiguate cases with greater target cardinality when there are multiple mappings to a parent target

    targetPath match {

      case Array(y) =>
        if (y.isObject() || y.isAttribute()) {
          val sourceHasArray = sourcePath.exists(x => x.isArrayOrRepeated()) //cardinality of target is greater
          if (sourceHasArray) {
            val nextRepeated = fromNextRepeated(sourcePath)
            val innerMapping = mapping.findOrCreateInnerMapping(nextRepeated.head, y)
            applyArrowAssignment(nextRepeated.tail, Array(y), innerMapping, sourceType, targetType)
          } else {
            mapping.addArrow(sourcePath.last, y, sourceType, targetType)
          }
        } else if (y.isArrayOrRepeated()) {
          mapping.findOrCreateInnerMapping(sourcePath.last, y)
        } else {
          throw new RuntimeException("Don't know how to assign type: " + y._type)
        }
      case htail =>
        val y = htail.head
        //only allow targets with equal or greater cardinality than source
        if (y.isArrayOrRepeated()) {
          val nextRepeated = fromNextRepeated(sourcePath)
          val innerMapping = mapping.findOrCreateInnerMapping(nextRepeated.head, y)
          applyArrowAssignment(nextRepeated.tail, htail.tail, innerMapping, sourceType, targetType)
        } else {
          //advance targetPath
          applyArrowAssignment(sourcePath, htail.tail, mapping, sourceType, targetType)
        }
    }
  }

  /**
    * Discard parts until a repetitive element is found
    * The first element of the returned seq is going to be a repeated element
    */
  private def fromNextRepeated(path: Array[NamePathElement]): Array[NamePathElement] = {
    val index = path.indexWhere(x => x.isRepeated() || x.isArray())
    if (index == -1) {
      throw new RuntimeException("No repetitive element was found in target path. Invalid assignment")
    }
    path.slice(index, path.length)
  }

  /**
    * Indicates if this source can be assigned
    *
    * @param source The source element
    * @param target The target element
    * @return
    */
  def canAssign(source: NamePathElement, target: NamePathElement): Boolean = {
    val c1 = source.cardinality
    val c2 = target.cardinality
    (c1 <= c2) && canAssignType(source._type, target._type)
  }

  private def canAssignType(sourceType: QNameType, targetType: QNameType) = {
    import QNameType._
    sourceType match {
      case Object | Attribute =>
        targetType == Object || targetType == Attribute
      case Repeated | Array =>
        targetType == Repeated || targetType == Array
    }
  }

}

/**
  * Sorts the assignments with the following criteria:
  * 1. cardinality of target (ascending)
  * 2. cardinality of source (descending)
  * 3. Insertion order
  *
  * This ordering ensures that when building the Mapping if target cardinality is greater
  * than source, the necessary parent mappings (if any) are already created.
  */
class ArrowComparator extends Comparator[(Int, FieldAssignment)] {
  override def compare(o1: (Int, FieldAssignment), o2: (Int, FieldAssignment)): Int = {
    val targetCardinality = Integer.compare(o1._2.target.cardinality, o2._2.target.cardinality)
    if (targetCardinality != 0) {
      return targetCardinality
    }

    val sourceCardinality = Integer.compare(o1._2.source.cardinality, o2._2.source.cardinality)
    if (sourceCardinality != 0) {
      return -sourceCardinality
    }

    o1._1.compareTo(o2._1)
  }
}