package org.mule.weave.v2.scaffolding

import org.mule.weave.v2.codegen.CodeWriter
import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.scaffolding.RandomPicker.pickRandom
import org.mule.weave.v2.scaffolding.RandomPicker.randomInt
import org.mule.weave.v2.scaffolding.RandomPicker.randomTwoDigitIntString
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.BinaryType
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.DateTimeType
import org.mule.weave.v2.ts.DynamicReturnType
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.LocalDateTimeType
import org.mule.weave.v2.ts.LocalDateType
import org.mule.weave.v2.ts.LocalTimeType
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NamespaceType
import org.mule.weave.v2.ts.NothingType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.PeriodType
import org.mule.weave.v2.ts.RangeType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.RegexType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TimeType
import org.mule.weave.v2.ts.TimeZoneType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.TypeType
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.UriType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.StringEscapeHelper

import scala.collection.JavaConverters
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.util.Random

/**
  * This service is going to generate a DataWeave scripting code that is being accepted by the specified WeaveType
  */
class ScaffoldingService {

  def scaffold(weaveType: WeaveType, mimeType: String, writerOptions: java.util.Map[String, Any], config: ScaffoldingConfiguration): String = {
    scaffold(weaveType, mimeType, writerOptions, config, NoScaffoldingFilter)
  }

  def scaffold(weaveType: WeaveType, mimeType: String, writerOptions: java.util.Map[String, Any], config: ScaffoldingConfiguration, filter: ScaffoldingFilter): String = {
    scaffold(weaveType, mimeType, JavaConverters.mapAsScalaMapConverter(writerOptions).asScala.toMap, config, filter)
  }

  def scaffold(weaveType: WeaveType, mimeType: String, writerOptions: Map[String, Any], config: ScaffoldingConfiguration): String = {
    scaffold(weaveType, mimeType, writerOptions, config, NoScaffoldingFilter)
  }

  def scaffold(weaveType: WeaveType, mimeType: String, writerOptions: Map[String, Any], config: ScaffoldingConfiguration, filter: ScaffoldingFilter): String = {
    val body: CodeWriter = new StringCodeWriter()
    val header: CodeWriter = new StringCodeWriter()
    val namespaces = ArrayBuffer[NamespaceDef]()
    doScaffold(weaveType, new ScaffolderRecursionDetector(), namespaces, body, DefaultDataGenerator, config, filter, WeaveTypePath(Seq()))
    header.println("%dw 2.0")
    if (namespaces.nonEmpty) {
      namespaces.foreach((nd) => {
        header.println(s"ns ${nd.prefix} ${nd.uri}")
      })
      header.println()
    }
    header.print(s"output ${mimeType} ")
    var first = true
    writerOptions.foreach((entry) => {
      if (!first) {
        header.print(", ")
      }
      header.print(entry._1).print("=")
      entry._2 match {
        case s: String  => header.print(StringEscapeHelper.escapeString(s))
        case b: Boolean => header.print(b.toString)
        case n: Number  => header.print(n.toString)
        case d          => header.print(StringEscapeHelper.escapeString(d.toString))
      }
      first = false
    })
    s"""
       |${header.toString}
       |---
       |${body.toString}
       |""".stripMargin

  }

  private def doScaffold(weaveType: WeaveType, recursionDetector: ScaffolderRecursionDetector, namespaces: ArrayBuffer[NamespaceDef], printer: CodeWriter, generator: SampleDataGenerator, config: ScaffoldingConfiguration, filter: ScaffoldingFilter, elementPath: WeaveTypePath): Unit = {
    weaveType match {
      case ObjectType(properties, _, _) => {
        printer.println("{")
        printer.indent()
        properties.foreach((prop) => {
          val qname = prop.key match {
            case KeyType(NameType(Some(qname)), _) => qname
            case _                                 => QName("_")
          }
          val path = elementPath.child(FieldTypeElement(qname))
          if (filter.accepts(path, prop.value)) {
            var i = 0
            val amount = if (prop.repeated) config.repeatedElementsAmount else 1
            while (i < amount) {
              doScaffold(prop.key, recursionDetector, namespaces, printer, generator, config, filter, path)
              printer.print(": ")
              doScaffold(prop.value, recursionDetector, namespaces, printer, SampleDataGenerator.pick(qname.name), config, filter, path)
              printer.println(",")
              i = i + 1
            }
          }
        })
        printer.dedent()
        printer.print("}")
        val maybeClassMetadata = weaveType.getMetadataConstraint("class")
        if (maybeClassMetadata.isDefined) {
          printer
            .printSpace()
            .printSpace("as Object")
            .print("{ class: ")
            .print(StringEscapeHelper.escapeString(maybeClassMetadata.get.value.toString))
            .print("}")
        }
      }
      case t: ReferenceType => {
        recursionDetector.resolve(
          t,
          (wt) => {
            doScaffold(wt, recursionDetector, namespaces, printer, generator, config, filter, elementPath)
          }, {
            case _: ObjectType => printer.println("{}")
            case _: ArrayType  => printer.println("[]")
            case _             => printer.println("null")
          })
      }
      case KeyType(name, attrs) => {
        doScaffold(name, recursionDetector, namespaces, printer, generator, config, filter, elementPath)
        printer.printSpace()
        if (attrs.nonEmpty) {
          printer.print("@(")
          var first = true
          attrs.foreach((att) => {
            val qname = att.name match {
              case NameType(Some(qname)) => qname
              case _                     => QName("_")
            }
            val path = elementPath.child(AttributeTypeElement(qname))
            if (filter.accepts(path, att.value)) {
              if (!first) {
                printer.printSpace(",")
              }
              doScaffold(att.name, recursionDetector, namespaces, printer, generator, config, filter, path)
              printer.print(": ")
              doScaffold(att.value, recursionDetector, namespaces, printer, SampleDataGenerator.pick(qname.name), config, filter, path)
              first = false
            }
          })
          printer.printSpace(")")
        }
      }
      case NameType(value) => {
        value match {
          case Some(qname) => {
            qname.ns match {
              case Some(namespace) => {
                val prefix = s"ns${namespaces.size}"
                printer.print(prefix)
                printer.print("#")
                namespaces.+=(NamespaceDef(prefix, namespace))
              }
              case None =>
            }
            if (StringEscapeHelper.keyRequiresQuotes(qname.name)) {
              printer.print(StringEscapeHelper.escapeString(qname.name))
            } else {
              printer.print(qname.name)
            }
          }
          case None => printer.print("''")
        }
      }
      case ArrayType(of) => {
        var i = 0
        printer.print("[")
        val path = elementPath.child(ArrayTypeElement())
        if (filter.accepts(path, of)) {
          while (i < config.repeatedElementsAmount) {
            printer.indent()
            //Two items
            if (i > 0) {
              printer.println(",")
            }
            doScaffold(of, recursionDetector, namespaces, printer, generator, config, filter, path)
            printer.dedent()
            i = i + 1
          }

        }
        printer.print("]")

        val maybeClassMetadata = weaveType.getMetadataConstraint("class")
        if (maybeClassMetadata.isDefined) {
          val className = maybeClassMetadata.get.value.toString
          //If it is not the default class name
          if (!className.equals(classOf[java.util.ArrayList[_]].getName))
            printer
              .printSpace()
              .printSpace("as Array")
              .print("{ class: ")
              .print(StringEscapeHelper.escapeString(className))
              .print("}")
        }
      }
      case UnionType(of) => {
        val interestingTypes = of
          .filterNot(_.isInstanceOf[NullType])
        doScaffold(pickRandom(interestingTypes), recursionDetector, namespaces, printer, generator, config, filter, elementPath)
      }
      case IntersectionType(of) => {
        val weaveType = TypeHelper.resolveIntersection(of)
        weaveType match {
          case IntersectionType(of) =>
            // If there's an intersection, just choose one
            doScaffold(of.headOption.getOrElse(AnyType()), recursionDetector, namespaces, printer, generator, config, filter, elementPath)
          case otherType =>
            doScaffold(otherType, recursionDetector, namespaces, printer, generator, config, filter, elementPath)
        }
      }
      case StringType(Some(value)) => {
        printer.print(StringEscapeHelper.escapeString(value))
      }
      case StringType(None) => {
        printer.print(StringEscapeHelper.escapeString(generator.generateString()))
      }
      case NamespaceType(prefix, namespace) => {
        printer.print("''")
      }
      case AnyType() => {
        printer.print("{}")
      }
      case BooleanType(_, _) => {
        printer.print(pickRandom(Seq("true", "false")))
      }
      case NumberType(Some(value)) => {
        printer.print(value)
      }
      case NumberType(None) => {
        printer.print(generator.generateNumber().toString)
      }
      case RangeType() => {
        printer.print("1 to 10")
      }
      case UriType(value) => {
        val uriStr = value.getOrElse("http://acme.com")
        printer.print(StringEscapeHelper.escapeString(uriStr))
      }
      case DateTimeType() => {
        printer.print(s"|${randomInt(2000, 2020)}-${randomTwoDigitIntString(12)}-${randomTwoDigitIntString(28)}T${randomTwoDigitIntString(24)}:${randomTwoDigitIntString(60)}:${randomTwoDigitIntString(60)}Z|")
      }
      case LocalDateTimeType() => {
        printer.print(s"|${randomInt(2000, 2020)}-${randomTwoDigitIntString(12)}-${randomTwoDigitIntString(28)}T${randomTwoDigitIntString(24)}:${randomTwoDigitIntString(60)}:${randomTwoDigitIntString(60)}|")
      }
      case LocalDateType() => {
        printer.print(s"|${randomInt(2000, 2020)}-${randomTwoDigitIntString(12)}-${randomTwoDigitIntString(28)}|")
      }
      case LocalTimeType() => {
        printer.print(s"|${randomTwoDigitIntString(24)}:${randomTwoDigitIntString(60)}:${randomTwoDigitIntString(60)}|")
      }
      case TimeType() => {
        printer.print(s"|${randomTwoDigitIntString(24)}:${randomTwoDigitIntString(60)}:${randomTwoDigitIntString(60)}Z|")
      }
      case TimeZoneType() => {
        printer.print(s"''")
      }
      case PeriodType() => {
        printer.print(s"|P${randomInt(24)}DT${randomInt(24)}H|")
      }
      case BinaryType() => {
        printer.print(StringEscapeHelper.escapeString(generator.generateBinary()))
      }
      case TypeType(t) => {
        printer.print(s"'${t.label().getOrElse("String")}'")
      }
      case RegexType() =>
      case NullType() => {
        printer.print("null")
      }
      case NothingType()        =>
      case _: TypeParameter     =>
      case _: FunctionType      =>
      case _: DynamicReturnType =>
    }
  }

}

case class ScaffoldingConfiguration(repeatedElementsAmount: Int)

object RandomPicker {
  val random = new Random()
  def pickRandom[T](options: Seq[T]): T = {
    options.apply(random.nextInt(options.size))
  }

  def randomInt(base: Int, max: Int): Int = {
    val randomMax = max - base
    base + random.nextInt(randomMax)
  }

  def randomDouble(): Double = {
    random.nextDouble()
  }

  def randomInt(max: Int): Int = {
    randomInt(1, max)
  }

  def randomFloat(base: Float, max: Float): Double = {
    base + (random.nextFloat() * max)
  }

  def randomTwoDigitIntString(max: Int): String = {
    randomInt(1, max).formatted("%02d")
  }
}

case class NamespaceDef(prefix: String, uri: String)

class ScaffolderRecursionDetector {
  private val stack: mutable.Stack[WeaveType] = mutable.Stack()

  private def alreadyInStack(id: WeaveType): Boolean = {
    stack.exists(_ eq id)
  }

  def resolve(t: ReferenceType, callback: (WeaveType) => Unit, shortcitcuit: (WeaveType) => Unit): Unit = {
    val weaveType = t.resolveType()
    if (alreadyInStack(weaveType)) {
      shortcitcuit(weaveType)
    } else {
      push(weaveType)
      callback(weaveType)
      pop()
    }
  }

  def push(t: WeaveType): Unit = {
    stack.push(t)
  }

  def pop(): Unit = {
    stack.pop()
  }
}