package org.mule.weave.v2.utils

import scala.language.implicitConversions

object StringHelper {

  val abrevMarker: String = "..."

  def levenshtein(str1: String, str2: String): Int = {
    val lenStr1 = str1.length
    val lenStr2 = str2.length

    val d: Array[Array[Int]] = Array.ofDim(lenStr1 + 1, lenStr2 + 1)

    for (i <- 0 to lenStr1) d(i)(0) = i
    for (j <- 0 to lenStr2) d(0)(j) = j

    for (i <- 1 to lenStr1; j <- 1 to lenStr2) {
      val cost = if (str1(i - 1) == str2(j - 1)) 0 else 1

      d(i)(j) = min(
        d(i - 1)(j) + 1, // deletion
        d(i)(j - 1) + 1, // insertion
        d(i - 1)(j - 1) + cost // substitution
      )
    }

    d(lenStr1)(lenStr2)
  }

  def indent(content: String): String = {
    "\t" + content.replaceAll("\n", "\n\t")
  }

  def min(nums: Int*): Int = nums.min

  /**
    * <p>Abbreviates a String using ellipses. This will turn
    * "Now is the time for all good men" into "Now is the time for..."</p>
    *
    * <p>Specifically:</p>
    * <ul>
    * <li>If the number of characters in str is less than or equal to
    * maxWidth, return str.</li>
    * <li>Else abbreviate it to (substring(str, 0, max-3) + "...").</li>
    * <li>If maxWidth is less than 4, throw an
    * IllegalArgumentException.</li>
    * <li>In no case will it return a String of length greater than
    * maxWidth.</li>
    * </ul>
    *
    * <pre>
    * StringUtils.abbreviate(null, *)      = null
    * StringUtils.abbreviate("", 4)        = ""
    * StringUtils.abbreviate("abcdefg", 6) = "abc..."
    * StringUtils.abbreviate("abcdefg", 7) = "abcdefg"
    * StringUtils.abbreviate("abcdefg", 8) = "abcdefg"
    * StringUtils.abbreviate("abcdefg", 4) = "a..."
    * StringUtils.abbreviate("abcdefg", 3) = IllegalArgumentException
    * </pre>
    *
    * @param str      the String to check, may be null
    * @param maxWidth maximum length of result String, must be at least 4
    * @return abbreviated String, { @code null} if null String input
    * @throws IllegalArgumentException if the width is too small
    * @since 2.0
    */
  def abbreviate(str: String, maxWidth: Int): String = {
    var offset = 0
    if (str == null) {
      return null
    }
    if (maxWidth < 4) {
      throw new IllegalArgumentException("Minimum abbreviation width is 4")
    }
    if (str.length <= maxWidth) {
      return str
    }
    if (str.length < maxWidth - 3) {
      offset = str.length - (maxWidth - 3)
    }
    if (offset <= 4) {
      return str.substring(0, maxWidth - 3) + abrevMarker
    }
    if (maxWidth < 7) {
      throw new IllegalArgumentException("Minimum abbreviation width with offset is 7")
    }
    if (offset + maxWidth - 3 < str.length) {
      return abrevMarker + StringHelper.abbreviate(str.substring(offset), maxWidth - 3)
    }
    abrevMarker + str.substring(str.length - (maxWidth - 3))
  }

  def shorten(str: String, maxWidth: Int): String = {
    if (str == null) {
      return null
    }
    if (maxWidth < 8) {
      throw new IllegalArgumentException("Minimum abbreviation width is 4")
    }
    if (str.length <= maxWidth + 3) {
      return str
    }
    str.substring(0, (maxWidth / 2)) + abrevMarker + str.substring(str.length - (maxWidth / 2))
  }

  def rightPad(str: String, len: Int) = {
    str.padTo(len, " ").mkString
  }
  def leftPad(str: String, len: Int) = {
    s"%${len}s".format(str)
  }

  implicit def toStringTransformer(s: String): StringTransformer = {
    new StringTransformer(s)
  }

  class StringTransformer(str: String) {
    def stripMarginAndNormalizeEOL: String = {
      str.stripMargin.replace("\r\n", "\n")
    }
  }

}
