package org.mule.weave.v2.editor

import org.mule.weave.v2.editor.WeaveFunctionTestCase.DEFAULT_TEST_NAME
import org.mule.weave.v2.parser.annotation.TrailingCommaAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.functions.OverloadedFunctionNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.UnknownPosition
import org.mule.weave.v2.utils.StringHelper.toStringTransformer

import java.io.File

// This class is temporary, ideally we should have better abstractions for handling
// test suites
object WeaveUnitTestSuite {
  def initialTest(node: FunctionDirectiveNode, moduleName: String, testPath: Seq[String]): String = {
    createInitialTest(moduleName, createTest(testPath, node))
  }

  private def createInitialTest(moduleName: String, test: WeaveTest): String = {
    s"""%dw 2.0
       |import * from dw::test::Tests
       |import * from dw::test::Asserts
       |
       |import * from $moduleName
       |---
       |${test.getTest(0, lastComma = false)}""".stripMarginAndNormalizeEOL
  }

  private def createTest(testPath: Seq[String], functionNode: FunctionDirectiveNode): WeaveTest = {
    var result: WeaveTest = WeaveFunctionTestCase(functionNode)
    testPath.reverse foreach ((testCase) => {
      result = WeaveTestSuite(testCase, Seq(result))
    })
    result
  }

  private def getNewTestParams(testAst: AstNode, testPath: Seq[String]): (Option[Int], Seq[String], Boolean, Position, Int) = {
    var insertionLineMaybe: Option[Int] = None
    var needsCommaBefore: Boolean = false
    var commaPosition: Position = UnknownPosition
    var nextAst = testAst
    val newTestPath = testPath.dropWhile((testName) => {
      AstNodeHelper.exists(nextAst, {
        case FunctionCallNode(VariableReferenceNode(NameIdentifier("describedBy", _), _), FunctionCallParametersNode(Seq(StringNode(suiteName, _), array)), _, _) if suiteName == testName || testPath.head == testName => {
          nextAst = array
          insertionLineMaybe = Some(array.location().endPosition.line)
          if (array.children().nonEmpty && array.annotation(classOf[TrailingCommaAnnotation]).isEmpty) {
            needsCommaBefore = true
            commaPosition = array.children().last.location().endPosition
          }
          true
        }
        case _ => false
      })
    })
    var count = 0
    AstNodeHelper.traverseChildren(nextAst, {
      case FunctionCallNode(VariableReferenceNode(NameIdentifier("in", _), _), FunctionCallParametersNode(List(StringNode(suiteName, _), astNode)), _, _) if suiteName.matches(WeaveFunctionTestCase.DEFAULT_TEST_NAME_REGEX) => {
        val pattern = WeaveFunctionTestCase.DEFAULT_TEST_NAME_REGEX.r
        suiteName match {
          case pattern(x) =>
            if (x.isEmpty) {
              count = Math.max(count, 1)
            } else {
              count = Math.max(count, x.toInt)
            }
          case _ =>
          //no match
        }
        true
      }
      case _ => false
    })
    (insertionLineMaybe, newTestPath, needsCommaBefore, commaPosition, count)
  }

  def fromExisting(node: FunctionDirectiveNode, testAst: AstNode, testPath: Seq[String]): Option[WeaveUnitTestAddition] = {
    val originalSize: Int = testPath.size
    val (insertionLineMaybe, newTestPath, needsCommaBefore, commaPosition, count) = getNewTestParams(testAst, testPath)

    insertionLineMaybe.map((insertionLine) => {
      val test = createTest(newTestPath, node)
      WeaveUnitTestAddition(test.getTest(originalSize - newTestPath.size, lastComma = true, count), insertionLine, needsCommaBefore, commaPosition)
    })
  }

  def expectedPath(node: FunctionDirectiveNode, moduleName: String): FullTestPath = {
    val modPath: String = testFilePathFromNameIdentifier(moduleName)
    val testPath: Seq[String] = moduleName.split(NameIdentifier.SEPARATOR).last +: node.variable.name.split(NameIdentifier.SEPARATOR)

    FullTestPath(modPath, testPath)
  }

  def testFilePathFromNameIdentifier(moduleName: String): String = {
    moduleName.replace(NameIdentifier.SEPARATOR, File.separator) + "Test.dwl"
  }
}

trait WeaveTest {
  def getTest(indentLevel: Int, lastComma: Boolean = true, count: Int = 0): String
}

case class WeaveFunctionTestCase(astNode: FunctionDirectiveNode) extends WeaveTest {
  lazy val functionName: String = astNode.variable.name

  override def getTest(indentLevel: Int, lastComma: Boolean = true, count: Int = 0): String = {
    val literal = astNode.literal
    val numberParams = literal match {
      case fn: FunctionNode =>
        fn.params.paramList.size
      case ofn: OverloadedFunctionNode =>
        ofn.functions.map(_.params.paramList.size).max
      case _ => 1
    }

    val params = Seq.fill(numberParams)("???").mkString(", ")
    val indent = "    " * indentLevel
    val countStr = if (count > 0) s" (${count + 1})" else ""
    val test =
      s"""$indent"$DEFAULT_TEST_NAME$countStr" in do {
         |$indent    $functionName($params) must beObject()
         |$indent}${if (lastComma) "," else ""}
         |""".stripMarginAndNormalizeEOL
    test
  }
}

object WeaveFunctionTestCase {
  val DEFAULT_TEST_NAME: String = "It should do something"
  val DEFAULT_TEST_NAME_REGEX: String = (DEFAULT_TEST_NAME + """\s*\(?(\d*)\)?""")
}

case class WeaveTestSuite(suiteName: String, tests: Seq[WeaveTest]) extends WeaveTest {
  override def getTest(indentLevel: Int, lastComma: Boolean = true, count: Int = 0): String = {
    val indent = "    " * indentLevel
    s"""$indent"$suiteName" describedBy [
       |${tests.map(_.getTest(indentLevel + 1)).mkString("\n")}$indent]${if (lastComma) "," else ""}
       |""".stripMargin.stripMargin.stripMarginAndNormalizeEOL
  }
}

case class WeaveUnitTestAddition(test: String, replacementLine: Int, needsCommaBefore: Boolean = false, commaPosition: Position = UnknownPosition)

case class FullTestPath(filePath: String, testPath: Seq[String])
