package org.mule.weave.v2.grammar

import org.mule.weave.v2.grammar.literals.IntegerLiteral
import org.mule.weave.v2.grammar.literals.Literals
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.grammar.structure.Array
import org.mule.weave.v2.grammar.structure.Namespaces
import org.mule.weave.v2.grammar.structure.Object
import org.mule.weave.v2.parser.ErrorAstNode
import org.mule.weave.v2.parser.MissingFormatDefinition
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.UndefinedExpressionNode
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.types.NativeTypeNode
import org.mule.weave.v2.parser.ast.types.TypeParametersListNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.utils.DataWeaveVersion
import org.parboiled2.CharPredicate._
import org.parboiled2._

import scala.annotation.switch

trait Directives extends PositionTracking with Tokens with IntegerLiteral with Annotations with Namespaces with Literals with Variables with Functions with Object with Array {
  this: Directives with Grammar =>
  var documentSyntaxVersion: Option[DataWeaveVersion] = None

  val createMimeNode = (contentType: String) => {
    ContentType(contentType)
  }
  val createIdNode = (dataFormat: String) => {
    DataFormatId(dataFormat)
  }
  val createVersionDirective = (missingToken: Option[ErrorAstNode], major: VersionMajor, missingDot: Option[ErrorAstNode], minor: VersionMinor) => {
    val errors: Seq[ErrorAstNode] = (missingToken.toSeq ++ Seq(major) ++ missingDot.toSeq ++ Seq(minor))
      .filter(_.isInstanceOf[ErrorAstNode])
      .map(_.asInstanceOf[ErrorAstNode])

    if (missingToken.isDefined || missingDot.isDefined)
      ErrorDirectiveNode(errors)
    else
      VersionDirective(major, minor)
  }
  val createOutputDirective = (wtype: Option[WeaveTypeNode], formatExpression: FormatExpression, formatId: Option[DataFormatId], options: Option[Seq[DirectiveOption]]) => {
    formatExpression match {
      case ContentType(_)  => OutputDirective(formatId, formatExpression.asInstanceOf[ContentType], options, wtype)
      case DataFormatId(_) => OutputDirective(formatExpression.asInstanceOf[DataFormatId], options, wtype)
      case _               => OutputDirective(formatExpression.asInstanceOf[MissingFormatDefinition], options, wtype)
    }
  }

  val createInputDirective = (variable: NameIdentifier, wtype: Option[WeaveTypeNode], formatExpression: Option[FormatExpression], options: Option[Seq[DirectiveOption]]) => {
    formatExpression match {
      case Some(ct: ContentType)   => InputDirective(variable, ct, options, wtype)
      case Some(dfi: DataFormatId) => InputDirective(variable, dfi, options, wtype)
      case _                       => new InputDirective(variable, None, None, options, wtype)
    }
  }
  val createVarDirective = (variable: NameIdentifier, wtype: Option[WeaveTypeNode], literal: AstNode) => {
    VarDirective(variable, literal, wtype)
  }

  val createTypeDirective = (variable: NameIdentifier, typeParameters: Option[TypeParametersListNode], t: AstNode) => {
    val errors = Seq(variable, t).filter(_.isInstanceOf[ErrorAstNode]).map(_.asInstanceOf[ErrorAstNode])

    if (errors.nonEmpty) {
      ErrorDirectiveNode(errors)
    } else if (t.isInstanceOf[UndefinedExpressionNode]) {
      TypeDirective(variable, typeParameters, NativeTypeNode(variable.name))
    } else {
      TypeDirective(variable, typeParameters, t.asInstanceOf[WeaveTypeNode])
    }
  }

  val createFunctionDirective = (variable: NameIdentifier, literal: AstNode) => {
    FunctionDirectiveNode(variable, literal)
  }

  val createAnnotationDirective = (nameIdentifier: NameIdentifier, params: AnnotationParametersNode) => {
    AnnotationDirectiveNode(nameIdentifier, params)
  }

  val createAnnotationParametersNode = (params: Seq[AnnotationParameterNode]) => {
    AnnotationParametersNode(params)
  }

  val createAnnotationParameterNode = (name: NameIdentifier, weaveTypeNode: WeaveTypeNode, defaultValue: Option[AstNode]) => {
    AnnotationParameterNode(name, weaveTypeNode, defaultValue)
  }

  val createImportWithSubElementsDirective = (subElements: ImportedElements, importedModule: ImportedElement) => {
    ImportDirective(importedModule, subElements)
  }

  val createImportModuleDirective = (importedModule: ImportedElement) => {
    ImportDirective(importedModule)
  }

  val createImportedElements = (elements: Seq[ImportedElement]) => {
    ImportedElements(elements)
  }

  val createImportedElement = (name: NameIdentifier, alias: Option[NameIdentifier]) => {
    ImportedElement(name, alias)
  }

  val createImportedElementNoAlias = (name: NameIdentifier) => {
    ImportedElement(name)
  }

  val createOptionNode = (name: DirectiveOptionName, literal: AstNode) => {
    DirectiveOption(name, literal)
  }

  val createOptionNameNode = (name: String) => {
    DirectiveOptionName(name)
  }

  def directives: Rule1[Seq[DirectiveNode]] = rule {
    oneOrMore(directive).separatedBy(ws)
  }

  private def directive: Rule1[DirectiveNode] = rule {
    pushPosition ~ (annotations ~ ws ~ (directiveSwitch) ~> injectAnnotations) ~ injectPosition
  }

  private def directiveSwitch = rule {
    run {
      (cursorChar: @switch) match {
        case '%' => versionDirective
        case 'n' => namespaceDirective
        case 'v' => varDirective
        case 't' => typeDirective
        case 'f' => functionDirective
        case 'o' => outputDirective
        case 'i' => inputDirective | importDirective
        case 'a' => annotationDirective
        case _   => MISMATCH
      }
    }
  }

  def versionMajor: Rule1[VersionMajor] = namedRule("versionMajor") {
    atomic(pushPosition ~ (capture(integer) ~> createVersionMajorNode) ~ injectPosition)
  }

  val createVersionMajorNode = (n: String) => {
    VersionMajor(n)
  }

  def versionMinor: Rule1[VersionMinor] = namedRule("versionMinor") {
    atomic(pushPosition ~ (capture(integer) ~> createVersionMinorNode) ~ injectPosition)
  }

  val createVersionMinorNode = (n: String) => {
    VersionMinor(n)
  }

  def versionDirective: Rule1[DirectiveNode] = namedRule("%dw") {
    (versionDirectiveName ~!~
      (versionMajor | missingVersionMajor()) ~
      (ch('.') ~ push(None) | (missingToken("Missing version separator i.e. %dw 2.0", ".") ~> (a => Some(a)))) ~
      (versionMinor | missingVersionMinor())) ~> createVersionDirective
  }

  def outputDirective: Rule1[OutputDirective] = namedRule("output") {
    outputDirectiveName ~!~
      optional(objFieldSep ~ typeExpression ~ fws) ~
      (contentType ~ optional(withKeyword ~!~ (dataFormat | missingDataFormatDefinition("Missing output definition i.e. output application/csv with octet-stream")))
        | dataFormat ~ push(None)
        | missingFormatDefinition("Missing output definition i.e. output application/json or output json") ~ push(None)) ~!~
        optional(fws ~ options) ~> createOutputDirective
  }

  def typeDirective: Rule1[DirectiveNode] = {
    def typeValue() = rule {
      (typeExpression | undefinedExpression | missingTypeExpression())
    }
    namedRule("type") {
      typeDirectiveName ~!~
        (typeDeclaration | missingIdentifier("Missing Type identifier i.e. type MyString = String")) ~
        ws ~
        optional(typeParametersList) ~
        ws ~
        guardedByToken(() => assignment, typeValue, "=") ~> createTypeDirective
    }
  }

  def inputDirective: Rule1[InputDirective] = namedRule("input") {
    inputDirectiveName ~!~
      (variableDeclaration | invalidNameDeclaration | missingIdentifier("Missing input name. i.e input payload json")) ~
      optional(ws ~ objFieldSep ~!~ (typeExpression | missingTypeExpression())) ~
      fws ~
      optional(contentType | dataFormat) ~
      optional(fws ~ options) ~> createInputDirective
  }

  def varDirective: Rule1[VarDirective] = namedRule("var") {
    varDirectiveName ~!~
      (variableDeclaration | invalidNameDeclaration | missingIdentifier("Missing variable name. i.e var a = 1")) ~
      optional(ws ~ objFieldSep ~!~ (typeExpression | missingTypeExpression)) ~
      ws ~
      ((assignment ~!~ ws ~ (expr | missingExpression("Missing Variable Expression i.e var a = 1"))) | missingExpression("Missing Assignment and the Variable Expression i.e var a = 1")) ~> createVarDirective
  }

  def functionDirective: Rule1[FunctionDirectiveNode] = namedRule("fun") {
    functionDirectiveName ~!~
      (variableDeclaration | invalidNameDeclaration | missingIdentifier("Missing function name. i.e fun a() = 1")) ~!~
      ws ~
      (functionLiteral | missingFunctionExpression("Missing Function Body. e.g fun test(name: String) = name")) ~> createFunctionDirective
  }

  def importDirective: Rule1[ImportDirective] = namedRule("import") {
    importDirectiveName ~!~ (importWithSubElements | importModuleOnly)
  }

  def annotationDirective: Rule1[AnnotationDirectiveNode] = rule {
    annotationDirectiveName ~!~
      (nameIdentifierNode | invalidNameDeclaration | missingIdentifier("Missing annotation name. i.e annotation foo()")) ~
      ws ~
      annotationParameters ~> createAnnotationDirective
  }

  def annotationParameters: Rule1[AnnotationParametersNode] = rule {
    pushPosition ~ (parenStart ~ zeroOrMore(annotationParameter).separatedBy(commaSep) ~ (parenEnd | fail("')' for annotation declaration.")) ~> createAnnotationParametersNode) ~ injectPosition
  }

  def annotationParameter: Rule1[AnnotationParameterNode] = rule {
    pushPosition ~ (nameIdentifierNode ~ ws ~ objFieldSep ~ ws ~ typeExpression ~ optional(ws ~ variableInitialValue) ~> createAnnotationParameterNode) ~ injectPosition
  }

  def importWithSubElements: Rule1[ImportDirective] = rule {
    pushPosition ~ (importedElements ~ fws ~ fromKeyword ~ fws ~ importedModuleNoAlias) ~> createImportWithSubElementsDirective ~ injectPosition
  }

  def importedModuleNoAlias: Rule1[ImportedElement] = rule {
    (namedRef | invalidNameDeclaration | missingIdentifier("Missing module name. i.e import charCode from dw::core::String")) ~> createImportedElementNoAlias
  }

  def importedElements: Rule1[ImportedElements] = rule {
    pushPosition ~ (oneOrMore(importedElement).separatedBy(commaSep) ~> createImportedElements) ~ injectPosition
  }

  def importedElement: Rule1[ImportedElement] = rule {
    pushPosition ~ (((startImport ~ push(None)) | (namedVariable ~ optional(fws ~ asKeyword ~!~ (namedVariable | missingIdentifier("Missing imported element alias name. i.e import charCode as cc from dw::core::Strings"))))) ~> createImportedElement) ~ injectPosition
  }

  def importModuleOnly: Rule1[ImportDirective] = rule {
    pushPosition ~!~ importModule ~> createImportModuleDirective ~ injectPosition
  }

  def importModule: Rule1[ImportedElement] = rule {
    pushPosition ~ ((namedRef | invalidNameDeclaration | missingIdentifier("Missing import module name. i.e import dw::core::Strings")) ~ optional(fws ~ asKeyword ~!~ (namedVariable | missingIdentifier("missing import alias name. i.e import dw::core::Strings as DWStrings"))) ~> createImportedElement) ~ injectPosition
  }

  def startImport: Rule1[NameIdentifier] = namedRule("Star Import") {
    pushPosition ~ (str("*") ~ push("*") ~> createNameIdentifier) ~ injectPosition
  }

  def options: Rule1[Seq[DirectiveOption]] = rule {
    oneOrMore(option).separatedBy(commaSep) ~ optional(commaSep)
  }

  def option: Rule1[DirectiveOption] = {
    def value: Rule1[AstNode] = rule { literal | missingExpression("Missing directive expression. i.e output application/csv header=true") }

    rule {
      pushPosition ~ ((optionName ~!~ ws ~
        guardedByToken(() => assignment, () => value, "=", Some("Missing assignment token (=).  i.e output application/csv header=true"))) ~> createOptionNode) ~ injectPosition
    }
  }

  def optionName: Rule1[DirectiveOptionName] = rule {
    pushPosition ~ (atomic(clearSB() ~ AlphaNum ~ appendSB() ~ zeroOrMore(AlphaNum ~ appendSB()) ~ push(sb.toString.trim) ~ notAKeyword) ~> createOptionNameNode) ~ injectPosition
  }

  private def contentType: Rule1[ContentType] = rule {
    pushPosition ~ (mime ~ notAKeyword ~> createMimeNode) ~ injectPosition
  }

  private def mime: Rule1[String] = rule {
    atomic(
      (clearSB() ~ oneOrMore(AlphaNum ~ appendSB()) ~
        ch('/') ~ appendSB()
        ~ oneOrMore(AlphaNum ~ appendSB())
        ~ zeroOrMore((ch('-') | ch('+') | ch('.') | ch('_')) ~ appendSB() ~ oneOrMore(AlphaNum ~ appendSB()))) ~ push(sb.toString.trim))
  }

  private def dataFormat: Rule1[DataFormatId] = rule {
    pushPosition ~ (id ~ notAKeyword ~> createIdNode) ~ injectPosition
  }

  private def id: Rule1[String] = rule {
    atomic(clearSB() ~ oneOrMore(AlphaNum ~ appendSB()) ~ push(sb.toString.trim))
  }

  /**
    * Its intended use is to parse the syntax directory at the top of the script (barring comments)
    * and set the grammar version to choose the syntax. This rule only performs side effects so it
    * should always be used enclosed by the & operand that resets the parser to its original state after it finishes.
    * Not doing it will most likely break the parser as the consumed stream won't have its associated AST nodes generated
    *
    * If no directive is found it defaults to dw 2.0
    */
  def setSyntaxVersion = namedRule("setSyntaxVersion") {
    ws ~ (versionDirective | push(new VersionDirective())) ~> ((maybevd: DirectiveNode) => {
      val vd = maybevd match {
        case vd: VersionDirective => vd
        case _                    => new VersionDirective()
      }
      documentSyntaxVersion = Some(DataWeaveVersion(vd.major.v.toInt, vd.minor.v.toInt))
    })
  }
}
