package org.mule.weave.v2.grammar.literals

import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.grammar.Identifiers
import org.mule.weave.v2.grammar.Tokens
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.grammar.structure.Namespaces
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.types.IntersectionTypeNode
import org.mule.weave.v2.parser.ast.types._
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.RedefiningSchemaException
import org.mule.weave.v2.parser.location
import org.mule.weave.v2.parser.location.ParserPosition
import org.parboiled2._
import org.parboiled2.support.hlist.::
import org.parboiled2.support.hlist.HNil

trait TypeLiteral extends PositionTracking with Tokens with Identifiers with StringLiteral with Namespaces {
  this: TypeLiteral with Grammar =>

  def schema: Rule1[SchemaNode]

  def typeReferenceLiteral: Rule1[TypeReferenceNode] = rule {
    pushPosition ~ (namedRef ~ optional(ws ~ ch('<') ~ ws ~ oneOrMore(typeExpression).separatedBy(commaSep) ~ ws ~ ch('>')) ~ optional(ws ~ schema) ~> createTypeReferenceNode) ~ injectPosition
  }

  def typeReferenceSelectorLiteral: Rule1[WeaveTypeNodeWithSchema] = rule {
    typeReferenceLiteral ~ optional(typeFieldSelectors)
  }

  def typeFieldSelectors: Rule[WeaveTypeNodeWithSchema :: HNil, WeaveTypeNodeWithSchema :: HNil] = namedRule("Type Reference Selector e.g `myType.fieldName`") {
    oneOrMore(typeFieldSelector)
  }

  def typeFieldSelector: Rule[WeaveTypeNodeWithSchema :: HNil, WeaveTypeNodeWithSchema :: HNil] = namedRule("Type Reference Selector e.g `myType.fieldName`") {
    pushPosition ~ (dot ~ fieldName ~ optional(ws ~ schema) ~> createTypeReferenceSelectorNode) ~ injectPosition
  }

  def typeDeclaration: Rule1[NameIdentifier] = rule {
    namedVariable
  }

  def dynamicReturnTypeExpression: Rule1[WeaveTypeNode] = namedRule("?") {
    ch('?') ~> createDynamicReturnType
  }

  def typeExpression: Rule1[WeaveTypeNodeWithSchema] = namedRule("Type Expression. e.g: \nSimple: String\nUnion Type: String | Number\nObject: {a: Number}") {
    pushPosition ~ (annotations ~ unionType ~> injectAnnotationToWeaveType) ~ injectPosition
  }

  def unionType: Rule1[WeaveTypeNodeWithSchema] = rule {
    intersectionType ~ optional(oneOrMore(unionTypeExpression | metadataInjectorExpression).separatedBy(ws))
  }

  def intersectionType: Rule1[WeaveTypeNodeWithSchema] = rule {
    basicTypeExpression ~ optional(oneOrMore(intersectionTypeExpression))
  }

  def unionTypeExpression = namedRule("|") {
    (pipe ~ intersectionType ~> createUnionExpression) ~ injectPosition
  }

  def metadataInjectorExpression = namedRule("<~") {
    ((ws ~ metadataInjector ~!~ (schema | missingSchema())) ~> createMetadataTypeExpression) ~ injectPosition
  }

  def intersectionTypeExpression = namedRule("&") {
    (amp ~ basicTypeExpression ~> createIntersectionExpression) ~ injectPosition
  }

  def enclosedTypeExpression: Rule1[WeaveTypeNodeWithSchema] = rule {
    (pushPosition ~ parenStart ~ typeExpression ~ parenEnd) ~> markEnclosedTypeNode ~ optional(ws ~ schema) ~> checkSchemaDuplicatedDefinition
  }

  val markEnclosedTypeNode = (startPosition: ParserPosition, node: WeaveTypeNodeWithSchema) => {
    node.annotate(EnclosedMarkAnnotation(createWeaveLocation(startPosition)))
    node
  }

  def basicTypeExpression: Rule1[WeaveTypeNodeWithSchema] = rule {
    ws ~ (typeReferenceSelectorLiteral
      | objectTypeExpression
      | functionType
      | enclosedTypeExpression
      | literalType)
  }

  def literalType: Rule1[WeaveTypeNodeWithSchema] = rule {
    pushPosition ~ ((string | booleanLiteral | number) ~ optional(ws ~ schema) ~> createLiteralType) ~ injectPosition
  }

  def objectTypeExpression: Rule1[WeaveTypeNodeWithSchema] = namedRule("Object Type. e.g {a: Boolean}") {
    pushPosition ~ ((ch('{') ~!~ (exactObject | orderedObject | simpleObject) ~ ch('}')) ~ optional(ws ~ schema) ~> injectSchemaToWeaveType) ~ injectPosition
  }

  def simpleObject: Rule1[ObjectTypeNode] = namedRule("Object e.g {a: String}") {
    (ws ~ zeroOrMore(keyValuePairType).separatedBy(commaSep) ~ quiet(optional(commaSep)) ~ ws) ~> createSimpleObjectType
  }

  def exactObject: Rule1[ObjectTypeNode] = namedRule("Exact Object e.g {| a: String|}") {
    (ch('|') ~!~ simpleObject ~ ch('|')) ~> markCloseObject
  }

  def orderedObject: Rule1[ObjectTypeNode] = namedRule("Ordered Object") {
    (ch('-') ~!~ (exactObject | simpleObject) ~ ch('-')) ~> markOrderedObject
  }

  def keyValuePairType: Rule1[KeyValueTypeNode] = rule {
    pushPosition ~ (ws ~ ((parenStart ~ typeReferenceLiteral ~ parenEnd) | keyType) ~ optional(ws ~ ch('*') ~ ws ~ push(true)) ~ optional(ws ~ ch('?') ~ ws ~ push(true)) ~ ws ~ objFieldSep ~ typeExpression ~> createKeyValuePairType) ~ injectPosition
  }

  def keyType: Rule1[WeaveTypeNode] = rule {
    pushPosition ~ annotations ~ ws ~ ((nameWithPrefix | anyNameType) ~ optional(ws ~ attributesStart ~ zeroOrMore(nameValueType).separatedBy(commaSep) ~ quiet(optional(commaSep)) ~ attributesEnd) ~> createKeyType) ~ injectPosition
  }

  def anyNameType: Rule1[NameTypeNode] = rule {
    (pushPosition ~ ws ~ ch('_') ~> createAnyName) ~ injectPosition
  }

  def nameValueType: Rule1[NameValueTypeNode] = rule {
    pushPosition ~ ws ~ ((nameWithPrefix | anyNameType) ~ optional(ws ~ ch('?') ~ push(true)) ~ ws ~ objFieldSep ~ ws ~ typeExpression ~> createNameValueType) ~ injectPosition
  }

  def nameWithPrefix: Rule1[NameTypeNode] = rule {
    pushPosition ~ (annotations ~ ws ~ namespace ~ (nameIdentifier | namedString) ~> createNameWithPrefix) ~ injectPosition
  }

  def typeParameterType: Rule1[TypeParameterNode] = namedRule("Type Parameter") {
    pushPosition ~ (nameIdentifierNode ~ optional(baseTypeKeyword ~ typeExpression) ~> createTypeParameter) ~ injectPosition
  }

  def functionType: Rule1[WeaveTypeNodeWithSchema] = namedRule("Lambda Type. e.g (a: String) -> Boolean") {
    pushPosition ~ (parenStart ~ zeroOrMore(functionTypeParameter).separatedBy(commaSep) ~ (parenEnd) ~ atomic(ws ~ str("->") ~ ws) ~ typeExpression ~ optional(ws ~ schema) ~> createFunctionType) ~ injectPosition
  }

  def functionTypeParameter: Rule1[FunctionParameterTypeNode] = namedRule("Function Parameter Type. e.g (a: String) -> Boolean") {
    pushPosition ~ (functionTypeParameterName ~ typeExpression ~> createFunctionParameterTypeNode) ~ injectPosition
  }

  def functionTypeParameterName: Rule2[NameIdentifier, Boolean] = rule {
    (nameIdentifierNode ~ ((ch('?') ~ push(true)) | push(false)) ~ ws ~ objFieldSep) | (push(null.asInstanceOf[NameIdentifier]) ~ push(false))
  }

  def typeParametersList: Rule1[TypeParametersListNode] = namedRule("Type Parameter List. e.g  <T>(a: T) -> Boolean") {
    pushPosition ~ (ch('<') ~ ws ~ oneOrMore(typeParameterType).separatedBy(commaSep) ~ ws ~ ch('>') ~> createTypeParameterList) ~ injectPosition
  }

  def typeParameterApplicationList: Rule1[TypeParametersApplicationListNode] = namedRule("Type Parameter Application. e.g f<Boolean, Number>(true, 2)") {
    pushPosition ~ (ws ~ ch('<') ~ ws ~ zeroOrMore(typeExpression | missingTypeExpression()).separatedBy(commaSep) ~ ws ~ ch('>') ~> createTypeParameterApplicationList) ~ injectPosition
  }

  def namedString: Rule1[String] = rule {
    doubleQuotedNamedString | singleQuotedNamedString
  }

  private def doubleQuotedNamedString: Rule1[String] = rule {
    ch('"') ~!~ clearSB() ~ oneOrMore(doubleQuoteEscapedChar | (noneOf("\"") ~ appendSB())) ~ push(sb.toString) ~ ch('"')
  }

  private def singleQuotedNamedString: Rule1[String] = rule {
    ch('\'') ~!~ clearSB() ~ oneOrMore(singleQuoteEscapedChar | (noneOf("'") ~ appendSB())) ~ push(sb.toString) ~ ch('\'')
  }

  private def singleQuoteEscapedChar = rule {
    ch('\\') ~ anyOf("'") ~ appendSB('\\') ~ appendSB()
  }

  private def doubleQuoteEscapedChar = rule {
    ch('\\') ~ anyOf("\"") ~ appendSB('\\') ~ appendSB()
  }

  //  Factories
  val createLiteralType: (AstNode, Option[SchemaNode]) => LiteralTypeNode = (sn: AstNode, schema: Option[SchemaNode]) => LiteralTypeNode(sn, schema)

  val createTypeParameter: (NameIdentifier, Option[WeaveTypeNode]) => TypeParameterNode = (name: NameIdentifier, baseType: Option[WeaveTypeNode]) => TypeParameterNode(name, baseType)

  val createKeyValuePairType: (WeaveTypeNode, Option[Boolean], Option[Boolean], WeaveTypeNode) => KeyValueTypeNode = (key: WeaveTypeNode, repeated: Option[Boolean], conditional: Option[Boolean], value: WeaveTypeNode) =>
    KeyValueTypeNode(key, value, repeated.getOrElse(false), conditional.getOrElse(false))

  val injectSchemaToWeaveType: (WeaveTypeNodeWithSchema, Option[SchemaNode]) => WeaveTypeNodeWithSchema = (otn, schema) => schema.map(otn.withSchema).getOrElse(otn)

  val injectAnnotationToWeaveType: (Seq[AnnotationNode], WeaveTypeNodeWithSchema) => WeaveTypeNodeWithSchema = (annotations, otn) => {
    otn.setAnnotations(annotations)
    otn
  }

  val checkSchemaDuplicatedDefinition: (WeaveTypeNodeWithSchema, Option[SchemaNode]) => WeaveTypeNodeWithSchema = (otn, schema) => {
    if (otn.asSchema.isDefined && schema.isDefined) {
      throw new RedefiningSchemaException(schema.get.location())
    } else {
      injectSchemaToWeaveType(otn, schema)
    }
  }

  val markCloseObject: (ObjectTypeNode) => ObjectTypeNode = (otn) => otn.copy(close = true)

  val markOrderedObject: (ObjectTypeNode) => ObjectTypeNode = (otn) => otn.copy(ordered = true)

  val createDynamicReturnType: () => DynamicReturnTypeNode = () => DynamicReturnTypeNode()

  val createSimpleObjectType: (Seq[WeaveTypeNode]) => ObjectTypeNode = (properties: Seq[WeaveTypeNode]) => ObjectTypeNode(properties)

  val createFunctionType = (args: Seq[FunctionParameterTypeNode], returnType: WeaveTypeNode, schema: Option[SchemaNode]) => FunctionTypeNode(args, returnType, schema)

  val createFunctionParameterTypeNode: (NameIdentifier, Boolean, WeaveTypeNode) => FunctionParameterTypeNode = (name: NameIdentifier, optional: Boolean, valueType: WeaveTypeNode) => FunctionParameterTypeNode(Option(name), valueType, optional)

  val createKeyType: (Seq[AnnotationNode], NameTypeNode, Option[Seq[NameValueTypeNode]]) => KeyTypeNode = (annotations: Seq[AnnotationNode], name: NameTypeNode, attributes: Option[Seq[NameValueTypeNode]]) => KeyTypeNode(name, attributes.getOrElse(Seq()), annotations)

  val createNameValueType: (NameTypeNode, Option[Boolean], WeaveTypeNode) => NameValueTypeNode = (name: NameTypeNode, optional: Option[Boolean], value: WeaveTypeNode) => NameValueTypeNode(name, value, optional.getOrElse(false))

  val createAnyName: () => NameTypeNode = () => NameTypeNode(None)

  val createTypeReferenceNode: (NameIdentifier, Option[Seq[WeaveTypeNode]], Option[SchemaNode]) => TypeReferenceNode = (nameSlot: NameIdentifier, typeParameters: Option[Seq[WeaveTypeNode]], schema: Option[SchemaNode]) => TypeReferenceNode(nameSlot, typeParameters, schema)

  val createTypeReferenceSelectorNode: (WeaveTypeNodeWithSchema, location.ParserPosition, NameNode, Option[SchemaNode]) => org.parboiled2.support.hlist.::[location.Position, org.parboiled2.support.hlist.::[WeaveTypeNodeWithSchema, HNil]] = (weaveTypeNode: WeaveTypeNodeWithSchema, parserPosition: location.ParserPosition, nameSlot: NameNode, schema: Option[SchemaNode]) => parserPosition :: TypeSelectorNode(nameSlot, weaveTypeNode, schema) :: HNil

  val createNameWithPrefix: (Seq[AnnotationNode], Option[NamespaceNode], String) => NameTypeNode = (annotationNodes: Seq[AnnotationNode], prefix: Option[NamespaceNode], localName: String) => NameTypeNode(Some(localName), prefix, None, codeAnnotations = annotationNodes)

  val createTypeParameterList: (Seq[TypeParameterNode]) => TypeParametersListNode = (parameters: Seq[TypeParameterNode]) => TypeParametersListNode(parameters)

  val createTypeParameterApplicationList = (parameters: Seq[WeaveTypeNode]) => {
    TypeParametersApplicationListNode(parameters)
  }

  val createUnionExpression: (WeaveTypeNodeWithSchema, WeaveTypeNodeWithSchema) => org.parboiled2.support.hlist.::[location.Position, org.parboiled2.support.hlist.::[UnionTypeNode, HNil]] = (left: WeaveTypeNodeWithSchema, right: WeaveTypeNodeWithSchema) => {
    (left, right) match {
      case (l @ UnionTypeNode(lElems, None, None, Seq()), UnionTypeNode(rElems, None, None, Seq())) => {
        resolveStartPosition(l) :: UnionTypeNode(lElems ++ rElems, None, None) :: HNil
      }
      case (l @ UnionTypeNode(lElems, None, None, Seq()), right) => {
        resolveStartPosition(l) :: UnionTypeNode(lElems :+ right, None, None) :: HNil
      }
      case (l, UnionTypeNode(rElems, None, None, Seq())) => {
        resolveStartPosition(l) :: UnionTypeNode(Seq(l) ++ rElems, None, None) :: HNil
      }
      case _ => resolveStartPosition(left) :: UnionTypeNode(Seq(left, right)) :: HNil
    }
  }

  val createIntersectionExpression: (WeaveTypeNodeWithSchema, WeaveTypeNodeWithSchema) => org.parboiled2.support.hlist.::[location.Position, org.parboiled2.support.hlist.::[IntersectionTypeNode, HNil]] = (left: WeaveTypeNodeWithSchema, right: WeaveTypeNodeWithSchema) => {
    (left, right) match {
      case (l @ IntersectionTypeNode(lElems, None, None, Seq()), IntersectionTypeNode(rElems, None, None, Seq())) => {
        resolveStartPosition(l) :: IntersectionTypeNode(lElems ++ rElems, None, None) :: HNil
      }
      case (l @ IntersectionTypeNode(lElems, None, None, Seq()), right) => {
        resolveStartPosition(l) :: IntersectionTypeNode(lElems :+ right, None, None) :: HNil
      }
      case (l, IntersectionTypeNode(rElems, None, None, Seq())) => {
        resolveStartPosition(l) :: IntersectionTypeNode(Seq(l) ++ rElems, None, None) :: HNil
      }
      case _ => resolveStartPosition(left) :: IntersectionTypeNode(Seq(left, right)) :: HNil
    }
  }

  val createMetadataTypeExpression: (WeaveTypeNodeWithSchema, SchemaNode) => org.parboiled2.support.hlist.::[location.Position, org.parboiled2.support.hlist.::[WeaveTypeNodeWithSchema, HNil]] = (left: WeaveTypeNodeWithSchema, right: SchemaNode) => {
    resolveStartPosition(left) :: left.withTypeSchema(right) :: HNil
  }
}

object TypeLiteral {
  val STRING_TYPE_NAME: String = "String"
  val ARRAY_TYPE_NAME: String = "Array"
  val ANY_TYPE_NAME: String = "Any"
  val OBJECT_TYPE_NAME: String = "Object"
  val BOOLEAN_TYPE_NAME: String = "Boolean"
  val NUMBER_TYPE_NAME: String = "Number"
  val RANGE_TYPE_NAME: String = "Range"
  val REGEX_TYPE_NAME: String = "Regex"
  val URI_TYPE_NAME: String = "Uri"
  val DATE_TYPE_NAME: String = "Date"
  val TYPE_TYPE_NAME: String = "Type"
  val TIME_TYPE_NAME: String = "Time"
  val NOTHING_TYPE_NAME: String = "Nothing"
  val DATETIME_TYPE_NAME: String = DATE_TYPE_NAME + TIME_TYPE_NAME
  val LOCALDATETIME_TYPE_NAME: String = "Local" + DATETIME_TYPE_NAME
  val LOCALTIME_TYPE_NAME: String = "Local" + TIME_TYPE_NAME
  val TIMEZONE_TYPE_NAME: String = TIME_TYPE_NAME + "Zone"
  val PERIOD_TYPE_NAME: String = "Period"
  val BINARY_TYPE_NAME: String = "Binary"
  val NULL_TYPE_NAME: String = "Null"
  val KEY_TYPE_NAME: String = "Key"
  val KEY_VALUE_PAIR_TYPE_NAME: String = "KeyValuePair"
  val ATTRIBUTES_TYPE_NAME: String = "Attributes"
  val NAME_TYPE_NAME: String = "Name"
  val NAME_VALUE_PAIR_TYPE_NAME: String = "NameValuePair"
  val NAME_SPACE_TYPE_NAME: String = "Namespace"
  val SCHEMA_TYPE_NAME: String = "Schema"
  val SCHEMA_PROPERTY_TYPE_NAME: String = "SchemaProperty"
  val FUNCTION_TYPE_NAME: String = "Function"
  val TRAIT_TYPE_NAME: String = "Trait"

  def isSystemType(name: String): Boolean = {
    systemTypeNames.contains(name)
  }

  val systemTypeNames = Set(
    STRING_TYPE_NAME,
    ARRAY_TYPE_NAME,
    ANY_TYPE_NAME,
    OBJECT_TYPE_NAME,
    BOOLEAN_TYPE_NAME,
    NUMBER_TYPE_NAME,
    RANGE_TYPE_NAME,
    REGEX_TYPE_NAME,
    URI_TYPE_NAME,
    DATE_TYPE_NAME,
    TYPE_TYPE_NAME,
    TIME_TYPE_NAME,
    NOTHING_TYPE_NAME,
    DATETIME_TYPE_NAME,
    LOCALDATETIME_TYPE_NAME,
    LOCALTIME_TYPE_NAME,
    TIMEZONE_TYPE_NAME,
    PERIOD_TYPE_NAME,
    BINARY_TYPE_NAME,
    NULL_TYPE_NAME,
    KEY_TYPE_NAME,
    KEY_VALUE_PAIR_TYPE_NAME,
    ATTRIBUTES_TYPE_NAME,
    NAME_TYPE_NAME,
    NAME_VALUE_PAIR_TYPE_NAME,
    NAME_SPACE_TYPE_NAME,
    SCHEMA_TYPE_NAME,
    SCHEMA_PROPERTY_TYPE_NAME,
    FUNCTION_TYPE_NAME
  //TRAIT_TYPE_NAME
  )
}
