package org.mule.weave.v2.parser.ast.operators

import org.mule.weave.v2.api.tooling.ast.DWAstNodeKind
import org.mule.weave.v2.grammar.BinaryOpIdentifier
import org.mule.weave.v2.grammar.OpIdentifier
import org.mule.weave.v2.grammar.UnaryOpIdentifier
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.Children
import org.mule.weave.v2.parser.ast.ExpressionAstNode
import org.mule.weave.v2.parser.ast.MutableAstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode

sealed trait OpNode extends ExpressionAstNode {

  def opId: OpIdentifier

  override def toString: String = {
    opId.name
  }
}

case class UnaryOpNode(var opId: UnaryOpIdentifier, var rhs: AstNode, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends OpNode with MutableAstNode {

  override def children(): Seq[AstNode] = Children(rhs).++=(codeAnnotations).result()

  override protected def doClone(): AstNode = {
    copy(opId, rhs.cloneAst(), codeAnnotations.map(_.cloneAst()))
  }

  /**
    * Sets the annotations to this node
    *
    * @param annotations The list of annotations
    */
  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  /**
    * Replace the child node that is equals to the given node with the new one
    *
    * @param toBeReplaced The node to be replaced
    * @param withNode     The replacement
    */
  override def update(toBeReplaced: AstNode, withNode: AstNode): Unit = {
    if (rhs eq toBeReplaced) {
      rhs = withNode
    }
  }

  override def getKind(): String = DWAstNodeKind.UNARY_OP_NODE
}

case class BinaryOpNode(var binaryOpId: BinaryOpIdentifier, var lhs: AstNode, var rhs: AstNode, var codeAnnotations: Seq[AnnotationNode] = Seq()) extends OpNode with MutableAstNode {
  override def children(): Seq[AstNode] = {
    Children(lhs, rhs).++=(codeAnnotations).result()
  }

  override def opId: BinaryOpIdentifier = binaryOpId

  override protected def doClone(): AstNode = {
    copy(binaryOpId, lhs.cloneAst(), rhs.cloneAst(), codeAnnotations.map(_.cloneAst()))
  }

  override def update(toBeReplaced: AstNode, withNode: AstNode): Unit = {
    if (lhs eq toBeReplaced) {
      lhs = withNode
    } else if (rhs eq toBeReplaced) {
      rhs = withNode
    }
  }

  /**
    * Sets the annotations to this node
    *
    * @param annotations The list of annotations
    */
  override def setAnnotations(annotations: Seq[AnnotationNode]): Unit = {
    this.codeAnnotations = annotations
  }

  override def getKind(): String = DWAstNodeKind.BINARY_OP_NODE
}
