package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.WeaveResource

import scala.collection.mutable

class ParsingNotificationManager {

  private val listeners = mutable.ArrayBuffer[ParsingNotificationListener]()

  def endPhase(nameIdentifier: NameIdentifier, value: CompilationPhase[_, _], result: PhaseResult[_]): Unit = {
    listeners.foreach((l) => {
      l.endPhase(nameIdentifier, value, result)
    })
  }

  def startPhase(nameIdentifier: NameIdentifier, value: CompilationPhase[_, _]): Unit = {
    listeners.foreach((l) => {
      l.startPhase(nameIdentifier, value)
    })
  }

  def addListener(listener: ParsingNotificationListener): ParsingNotificationManager = {
    listeners.+=(listener)
    this
  }

  def endResolvingResource(nameIdentifier: NameIdentifier, result: Option[WeaveResource]): Unit = {
    listeners.foreach((l) => {
      l.endResolvingResource(nameIdentifier, result)
    })
  }

  def startResolvingResource(nameIdentifier: NameIdentifier): Unit = {
    listeners.foreach((l) => {
      l.startResolvingResource(nameIdentifier)
    })
  }

  def progress(): Unit = {
    listeners.foreach((l) => {
      l.progress()
    })
  }
}

trait ParsingNotificationListener {

  def endPhase(nameIdentifier: NameIdentifier, value: CompilationPhase[_, _], result: PhaseResult[_]): Unit

  def startPhase(nameIdentifier: NameIdentifier, value: CompilationPhase[_, _]): Unit

  def endResolvingResource(nameIdentifier: NameIdentifier, result: Option[WeaveResource]): Unit

  def startResolvingResource(nameIdentifier: NameIdentifier): Unit

  def progress(): Unit

}

class PrintlnParsingNotificationListener extends ParsingNotificationListener {

  val metrics: mutable.Map[String, mutable.Map[CompilationPhase[_, _], Long]] = mutable.Map()
  var indent: Int = 0

  override def endPhase(nameIdentifier: NameIdentifier, phase: CompilationPhase[_, _], result: PhaseResult[_]): Unit = {
    if (ignorePhase(phase))
      return
    indent = indent - 1
    metrics.get(nameIdentifier.name) match {
      case Some(metric) => {
        metric.get(phase) match {
          case Some(value) => {
            println(s"|${"--|" * indent}[${nameIdentifier}] - [${phase.getClass.getSimpleName}] took ${System.currentTimeMillis() - value}ms")
          }
          case None => {
            println(s"[${nameIdentifier}] - [${phase.getClass.getSimpleName}] No Metrics Found  for this Phase")
          }
        }
      }
      case None => {
        println(s"[${nameIdentifier}] - [${phase.getClass.getSimpleName}] No Metrics Found for this Element")
      }
    }

  }

  override def startPhase(nameIdentifier: NameIdentifier, value: CompilationPhase[_, _]): Unit = {
    if (ignorePhase(value))
      return
    metrics.getOrElseUpdate(nameIdentifier.name, mutable.Map()).put(value, System.currentTimeMillis())
    indent = indent + 1
  }

  private def ignorePhase(value: CompilationPhase[_, _]) = {
    value.isInstanceOf[CompositeCompilationPhase[_, _, _]] || value.isInstanceOf[EnrichedCompilationPhase[_, _]]
  }

  override def endResolvingResource(nameIdentifier: NameIdentifier, result: Option[WeaveResource]): Unit = {}

  override def startResolvingResource(nameIdentifier: NameIdentifier): Unit = {}

  override def progress(): Unit = {}
}