package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.structure.KeyValuePairNode
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.Edge
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NameValuePairType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.TypeCoercer
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.WeaveTypeResolver
import org.mule.weave.v2.ts.resolvers.KeyValuePairTypeResolver.selectKeyValuePair

import scala.collection.mutable

object ObjectTypeResolver extends WeaveTypeResolver {

  override def resolveExpectedType(node: TypeNode, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): Seq[(Edge, WeaveType)] = {
    if (incomingExpectedType.isDefined) {
      node.incomingEdges()
        .flatMap((edge) => {
          edge.source.astNode match {
            case _: KeyValuePairNode =>
              val maybeKeyValuePairType = selectKeyValuePair(edge.source, incomingExpectedType)
              if (maybeKeyValuePairType.isDefined) {
                Some((edge, maybeKeyValuePairType.get))
              } else {
                None
              }
            case _ => {
              val weaveType: WeaveType = incomingExpectedType.get
              val dynamicExpressionType: WeaveType = TypeHelper.unify(Seq(weaveType, ArrayType(weaveType), NullType()))
              Some((edge, dynamicExpressionType))
            }
          }
        })
    } else {
      Seq()
    }
  }

  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val weaveTypes: Seq[WeaveType] = node.incomingEdges().map(_.incomingType())
    val properties: Seq[KeyValuePairType] =
      weaveTypes.flatMap {
        case kv: KeyValuePairType =>
          Seq(kv)
        case expressionNode => {
          resolveProperties(expressionNode, ctx)
        }
      }

    val repeatedFieldsByName: Map[Option[QName], Seq[KeyValuePairType]] = properties.filter((kvp) => {
      kvp.key match {
        case KeyType(NameType(_), _) => true
        case _                       => false
      }
    }).groupBy((kvp) => {
      (kvp.key: @unchecked) match {
        case KeyType(NameType(name), _) => name
      }
    })
      .filter(_._2.size > 1)

    val repeatedFields = repeatedFieldsByName.mapValues((fields) => {
      val name = fields.head.key.asInstanceOf[KeyType].name
      val unifiedValue = TypeHelper.unify(fields.map(_.value))
      val allAttributes = fields.flatMap((kvp) => kvp.key match {
        case KeyType(_, attrs) => attrs
        case _                 => Seq()
      })
      val attributes: Seq[NameValuePairType] = allAttributes.groupBy(_.name).map((nvp) => {
        //Is Optional if one of the attributes is optional or is not present in any of the repeated entries
        val optional = nvp._2.exists(_.optional) || nvp._2.size != fields.size
        NameValuePairType(nvp._1, TypeHelper.unify(nvp._2.map(_.value)), optional)
      }).toSeq
      val required = fields.exists(!_.optional)
      KeyValuePairType(KeyType(name, attributes), unifiedValue, !required, repeated = true)
    })

    val alreadyMapped = mutable.ArrayBuffer[Option[QName]]()

    val reducedProperties = properties.flatMap((kvp) => (kvp: @unchecked) match {
      case KeyValuePairType(KeyType(NameType(name), _), _, _, _) => {
        if (repeatedFields.contains(name)) {
          if (alreadyMapped.contains(name)) {
            None
          } else {
            alreadyMapped.+=(name)
            repeatedFields.get(name)
          }
        } else {
          Some(kvp)
        }
      }
      case _ => Some(kvp)
    })

    val open = properties.exists((kvp) => {
      (kvp.key: @unchecked) match {
        case KeyType(NameType(None), _) => true
        case _                          => false
      }
    })
    Some(ObjectType(reducedProperties, !open))
  }

  def resolveProperties(expression: WeaveType, ctx: WeaveTypeResolutionContext): Seq[KeyValuePairType] = {
    expression match {
      case ArrayType(of) => {
        val keyValuePairTypes = resolveObjectProperties(of, ctx)
        keyValuePairTypes.map((prop) => {
          val pairType = prop.cloneType().asInstanceOf[KeyValuePairType]
          pairType.repeated = true //Mark as repeated
          pairType
        })
      }
      case UnionType(of)          => of.flatMap((t) => resolveProperties(t, ctx))
      case objectType: ObjectType => resolveObjectProperties(objectType, ctx)
      case rt: ReferenceType      => resolveProperties(rt.resolveType(), ctx)
      case _                      => Seq()
    }
  }

  def resolveObjectProperties(of: WeaveType, ctx: WeaveTypeResolutionContext): Seq[KeyValuePairType] = {
    of match {
      case ObjectType(props, _, _) => props
      case UnionType(unionOf)      => unionOf.flatMap((t) => resolveObjectProperties(t, ctx))
      case rt: ReferenceType       => resolveObjectProperties(rt.resolveType(), ctx)
      case _ => {
        //Trying to coerce to object
        TypeCoercer.coerce(of, ObjectType(), ctx) match {
          case Some(x) => x match {
            case ObjectType(props, _, _) => props
          }
          case None => Seq()
        }
      }
    }
  }

}
