package org.mule.weave.v2.interpreted.module.reader

import org.mule.weave.v2.interpreted.module.reader.exception.WeaveReaderException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.ObjectValueBuilder
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.header.directives.ContentType
import org.mule.weave.v2.parser.ast.header.directives.DirectiveOptionName
import org.mule.weave.v2.parser.ast.header.directives.VersionMajor
import org.mule.weave.v2.parser.ast.header.directives.VersionMinor
import org.mule.weave.v2.parser.ast.operators.OpNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ModuleLoaderManager
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceFactory

class AstWeaveParser(name: String, val sourceProvider: SourceProvider, moduleLoaderManager: ModuleLoaderManager)(implicit ctx: EvaluationContext) {

  private val inputStream = sourceProvider.asInputStream

  def toValue(astNode: AstNode): Value[_] = {
    val builder = new ObjectValueBuilder()
    builder
      .addPair("class", astNode.getClass.getSimpleName)

    astNode match {
      case contentType: ContentType                 => builder.addPair("value", contentType.mime)
      case lvnode: LiteralValueAstNode              => builder.addPair("value", lvnode.literalValue)
      case opNode: OpNode                           => builder.addPair("value", opNode.opId.getClass.getSimpleName)
      case versionMinor: VersionMinor               => builder.addPair("value", versionMinor.v)
      case versionMajor: VersionMajor               => builder.addPair("value", versionMajor.v)
      case directiveOptionName: DirectiveOptionName => builder.addPair("value", directiveOptionName.name)
      case nameIdentifier: NameIdentifier           => builder.addPair("value", nameIdentifier.name)
      case _                                        =>
    }
    val nodes: Seq[AstNode] = astNode.children()
    if (nodes.nonEmpty) {

      val children = ArrayValue(nodes.map((node) => {
        toValue(node)
      }))

      builder.addPair("children", children)
    }

    builder.build
  }

  def parse(): Value[_] = {
    val stream: WeaveResource = WeaveResourceFactory.fromInputStream(name, inputStream)
    val context = new ParsingContext(NameIdentifier(name), new MessageCollector, ModuleParsingPhasesManager(moduleLoaderManager), attachDocumentation = false)
    val value: PhaseResult[ParsingResult[AstNode]] = new DocumentParser().parse(stream, context)
    if (value.hasResult()) {
      toValue(value.getResult().astNode)
    } else {
      throw new WeaveReaderException("Problem while parsing file : " + value.messages().errorMessageString(), value.errorMessages().head._1)
    }

  }

}
