package org.mule.weave.v2.interpreted.module.reader

import org.mule.weave.v2.model.DefaultEvaluationContext
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.UserDefinedServicesProvider
import org.mule.weave.v2.model.service.ClasspathProtocolHandler
import org.mule.weave.v2.model.service.CompositeSecurityMangerService
import org.mule.weave.v2.model.service.DefaultSecurityManagerService
import org.mule.weave.v2.model.service.ProtocolUrlSourceProviderResolverService
import org.mule.weave.v2.model.service.SecurityManagerService
import org.mule.weave.v2.model.service.UrlProtocolHandler
import org.mule.weave.v2.model.service.UrlSourceProviderResolverService
import org.mule.weave.v2.model.service.WeaveRuntimePrivilege
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ModuleLoaderManager
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.runtime.exception.CompilationExecutionException
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceFactory

import scala.collection.mutable.ArrayBuffer

class DefaultWeaveParser(name: String, val sourceProvider: SourceProvider, moduleLoaderManager: ModuleLoaderManager, readerSettings: WeaveReaderSettings)(implicit ctx: EvaluationContext) {

  private val inputStream = sourceProvider.asInputStream

  def parse(): Value[_] = {
    val stream: WeaveResource = WeaveResourceFactory.fromInputStream(name, inputStream)
    val context = new ParsingContext(NameIdentifier(name), new MessageCollector, ModuleParsingPhasesManager(moduleLoaderManager), attachDocumentation = false)
    val compileResult = WeaveCompiler.compile(stream, context)
    if (compileResult.hasErrors() || compileResult.isEmpty()) {
      val errorMessages: (WeaveLocation, Message) = compileResult.errorMessages().head
      throw new CompilationExecutionException(errorMessages._1, errorMessages._2.message)
    }

    val privileges = new ArrayBuffer[WeaveRuntimePrivilege]()
    if (readerSettings.javaModule) {
      privileges ++= WeaveRuntimePrivilege.JAVA_PERMISSIONS
    }

    privileges ++= readerSettings.privileges.split(",").map((privilege) => new WeaveRuntimePrivilege(privilege))

    val servicesProvider = UserDefinedServicesProvider(
      Map(
        classOf[SecurityManagerService] -> new CompositeSecurityMangerService(ctx.serviceManager.securityManager, new DefaultSecurityManagerService(privileges.toArray)),
        classOf[UrlSourceProviderResolverService] -> new ProtocolUrlSourceProviderResolverService(if (readerSettings.externalResources) Seq(ClasspathProtocolHandler, UrlProtocolHandler) else Seq(ClasspathProtocolHandler))))

    val sm: ServiceManager = new ServiceManager(ctx.serviceManager.loggingService, serviceProvider = servicesProvider)
    val newCtx = DefaultEvaluationContext(serviceManager = sm)
    compileResult.getResult().executable.execute()(newCtx).materialize(newCtx)
  }

}
