package org.mule.weave.v2.interpreted.module.reader

import org.mule.weave.v2.core.util.IntervalExecutor
import org.mule.weave.v2.interpreted.module.WeaveDataFormat
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.option.BooleanModuleOption
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.module.option.Settings
import org.mule.weave.v2.module.option.StringModuleOption
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.reader.SourceProviderAwareReader
import org.mule.weave.v2.parser.phase.ModuleLoaderManager
import org.mule.weave.v2.sdk.ParsingContextFactory

import java.io.File
import java.io.InputStream
import java.nio.charset.Charset
import java.util.concurrent.TimeUnit

class WeaveReader(override val sourceProvider: SourceProvider, moduleLoaderManager: ModuleLoaderManager, readerSettings: WeaveReaderSettings = WeaveReaderSettings())(implicit ctx: EvaluationContext) extends Reader with SourceProviderAwareReader {

  override def dataFormat: Option[DataFormat[_, _]] = Some(new WeaveDataFormat)

  override protected def doRead(name: String): Value[_] = {
    if (readerSettings.astMode) {
      WeaveReader.logAstModeUsed()
      val weaveParser = new AstWeaveParser(name, sourceProvider, moduleLoaderManager)
      weaveParser.parse()
    } else if (readerSettings.onlyData) {
      val parser = new OnlyDataInMemoryWeaveParser(name, sourceProvider)
      parser.parse()
    } else {
      ctx.serviceManager.loggingService.logWarn("Using Weave Reader at Runtime May Cause Performance Issues. It is strongly advice to either use with `onlyData=true` or try other MimeType. This format was design for debugging and design only.")
      val parser = new DefaultWeaveParser(name, sourceProvider, moduleLoaderManager, readerSettings)
      parser.parse()
    }
  }

  override val settings: Settings = readerSettings
}

object WeaveReader {

  private val deferredLogger: IntervalExecutor = new IntervalExecutor(1, TimeUnit.DAYS)

  def logAstModeUsed()(implicit ctx: EvaluationContext): Unit = {
    deferredLogger.trigger(() => {
      ctx.serviceManager.loggingService.logWarn("Using AstMode is for internal use Only. This may be removed in the future")
    })
  }

  def apply(file: File, encoding: String)(implicit ctx: EvaluationContext): WeaveReader = {
    new WeaveReader(SourceProvider(file, Charset.forName(encoding)), ParsingContextFactory.createModuleLoaderManager())
  }

  def apply(file: InputStream, encoding: String)(implicit ctx: EvaluationContext): WeaveReader = {
    new WeaveReader(SourceProvider(file, Charset.forName(encoding)), ParsingContextFactory.createModuleLoaderManager())
  }

  def apply(content: String)(implicit ctx: EvaluationContext): WeaveReader = {
    new WeaveReader(SourceProvider(content), ParsingContextFactory.createModuleLoaderManager())
  }

  def apply(sourceProvider: SourceProvider)(implicit ctx: EvaluationContext): WeaveReader = {
    new WeaveReader(sourceProvider, ParsingContextFactory.createModuleLoaderManager())
  }

  def apply(source: SourceProvider, moduleLoaderManager: ModuleLoaderManager, readerSettings: WeaveReaderSettings)(implicit ctx: EvaluationContext): WeaveReader = new WeaveReader(source, moduleLoaderManager, readerSettings)
}

trait OnlyDataSettings extends Settings {
  var onlyData: Boolean = _

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions +
      BooleanModuleOption(WeaveReaderSettings.dataOnlyProp, defaultValue = defaultOnlyDataValue, descriptionUrl = "data-format/dw/onlyData.asciidoc")
  }

  def defaultOnlyDataValue: Boolean = false

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case WeaveReaderSettings.dataOnlyProp => {
        onlyData = value.asInstanceOf[Boolean]
      }
      case _ => super.writeSettingsValue(settingName, value)
    }
  }
}

class WeaveReaderSettings extends OnlyDataSettings {
  var externalResources: Boolean = _
  var javaModule: Boolean = _
  var astMode: Boolean = _
  var privileges: String = ""

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions() +
      BooleanModuleOption(WeaveReaderSettings.externalResources, defaultValue = false, descriptionUrl = "data-format/dw/externalResources.asciidoc").markAsSecurity() +
      BooleanModuleOption(WeaveReaderSettings.javaModule, defaultValue = false, descriptionUrl = "data-format/dw/javaModule.asciidoc").markAsSecurity() +
      BooleanModuleOption(WeaveReaderSettings.astMode, defaultValue = false, descriptionUrl = "data-format/dw/astMode.asciidoc").markAsInternal() +
      StringModuleOption(WeaveReaderSettings.privileges, defaultValue = "", descriptionUrl = "data-format/dw/privileges.asciidoc").markAsSecurity()

  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case WeaveReaderSettings.externalResources => {
        externalResources = value.asInstanceOf[Boolean]
      }
      case WeaveReaderSettings.javaModule => {
        javaModule = value.asInstanceOf[Boolean]
      }
      case WeaveReaderSettings.astMode => {
        astMode = value.asInstanceOf[Boolean]
      }
      case WeaveReaderSettings.privileges => {
        privileges = value.asInstanceOf[String]
      }
      case _ => super.writeSettingsValue(settingName, value)
    }
  }
}

object WeaveReaderSettings {

  val dataOnlyProp = "onlyData"
  val javaModule = "javaModule"
  val astMode = "astMode"
  val privileges = "privileges"
  val externalResources = "externalResources"

  def apply(): WeaveReaderSettings = new WeaveReaderSettings()
}
