package org.mule.weave.v2.interpreted.node

import org.mule.weave.v2.interpreted.node.executors.BinaryFunctionExecutor
import org.mule.weave.v2.interpreted.node.executors.BinaryOpExecutor
import org.mule.weave.v2.interpreted.node.executors.BinaryStaticOverloadedFunctionExecutor
import org.mule.weave.v2.interpreted.node.executors.TernaryFunctionExecutor
import org.mule.weave.v2.interpreted.node.executors.TernaryOverloadedStaticExecutor
import org.mule.weave.v2.interpreted.node.executors.UnaryFunctionExecutor
import org.mule.weave.v2.interpreted.node.executors.UnaryOpExecutor
import org.mule.weave.v2.interpreted.node.executors.UnaryOverloadedFunctionExecutor
import org.mule.weave.v2.interpreted.node.pattern.TypePatternNode
import org.mule.weave.v2.interpreted.node.structure.KeyValuePairNode
import org.mule.weave.v2.interpreted.node.structure.LiteralValueNode
import org.mule.weave.v2.interpreted.node.structure.OverloadedFunctionNode
import org.mule.weave.v2.interpreted.node.structure.function.FunctionParameterNode
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBinding
import org.mule.weave.v2.interpreted.node.structure.header.directives.VarDirective
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.NameValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.LocationCapable
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.UnknownLocation

import scala.collection.GenTraversableOnce
import scala.collection.mutable.ArrayBuffer

class AstWrapper(p: Product) extends ObjectValue {

  override def evaluate(implicit ctx: EvaluationContext): T = {
    val products = p.productIterator
    val it: Iterator[KeyValuePair] = toKeyValuePair(products)
    ObjectSeq(it)
  }

  private def toKeyValuePair(products: Iterator[Any]): Iterator[KeyValuePair] = {
    products.flatMap((v) => toMaybeKeyValuePair(v))
  }

  @scala.annotation.tailrec
  private def toMaybeKeyValuePair(v: Any): GenTraversableOnce[KeyValuePair] = {
    v match {
      case n: ExecutionNode => {
        Some(KeyValuePair(new AstKeyWrapper(n, None), AstWrapper(n)))
      }
      case iterable: Iterable[ExecutionNode] => {
        toKeyValuePair(iterable.toIterator)
      }
      case o: Option[_] => {
        if (o.isDefined) {
          toMaybeKeyValuePair(o.get)
        } else {
          None
        }
      }
      case bf: BinaryOpExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("name"), StringValue(bf.name))
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        builder += NameValuePair(NameValue("secondArgConstantType"), BooleanValue(bf.secondArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case bf: BinaryFunctionExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("name"), StringValue(bf.name))
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        builder += NameValuePair(NameValue("secondArgConstantType"), BooleanValue(bf.secondArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case bf: BinaryStaticOverloadedFunctionExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("name"), StringValue(bf.name))
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        builder += NameValuePair(NameValue("secondArgConstantType"), BooleanValue(bf.secondArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case bf: UnaryOverloadedFunctionExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("name"), StringValue(bf.name))
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case bf: UnaryOpExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("name"), StringValue(bf.name))
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case bf: UnaryFunctionExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("name"), StringValue(bf.name))
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case bf: TernaryOverloadedStaticExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        builder += NameValuePair(NameValue("secondArgConstantType"), BooleanValue(bf.secondArgConstantType))
        builder += NameValuePair(NameValue("thirdArgConstantType"), BooleanValue(bf.thirdArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case bf: TernaryFunctionExecutor => {
        val builder = new ArrayBuffer[NameValuePair]
        builder += NameValuePair(NameValue("firstArgConstantType"), BooleanValue(bf.firstArgConstantType))
        builder += NameValuePair(NameValue("secondArgConstantType"), BooleanValue(bf.secondArgConstantType))
        builder += NameValuePair(NameValue("thirdArgConstantType"), BooleanValue(bf.thirdArgConstantType))
        val keyValue = KeyValue(QualifiedName(toValidName(bf), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(bf)))
      }
      case eb: ExternalBinding => {
        val builder = new ArrayBuffer[NameValuePair]
        val materialize = NameValuePair(NameValue("materialize"), BooleanValue(eb.needsMaterialize))
        builder += materialize
        val streamCapable = NameValuePair(NameValue("streamCapable"), BooleanValue(eb.streamCapable))
        builder += streamCapable
        val keyValue = KeyValue(QualifiedName(toValidName(eb), None), Some(AttributesValue(builder.result())))
        Some(KeyValuePair(keyValue, AstWrapper(eb)))
      }
      case p: Product => {
        Some(KeyValuePair(KeyValue(toValidName(p)), AstWrapper(p)))
      }
      case _ => {
        None
      }
    }
  }

  private def toValidName(p: Product) = {
    val name = p.getClass.getSimpleName
    name.replace('$', '_')
  }

  override def location(): Location = p match {
    case lc: LocationCapable => lc.location()
    case _                   => UnknownLocation
  }
}

class AstKeyWrapper(node: ExecutionNode, actualTypes: Option[Seq[Type]]) extends KeyValue {

  override def location(): Location = node.location()

  override def attributes(implicit ctx: EvaluationContext): Option[AttributesValue] = {
    val builder = new ArrayBuffer[NameValuePair]
    node match {
      case vn: NameSlot => {
        val slot = NameValuePair(NameValue("slot"), NumberValue(vn.slot))
        builder += slot
        val name = NameValuePair(NameValue("name"), StringValue(vn.name))
        builder += name
      }
      case fpn: FunctionParameterNode => {
        val materialize = NameValuePair(NameValue("materialize"), BooleanValue(fpn.materialize))
        builder += materialize
        val typeRequiresMaterialize = NameValuePair(NameValue("typeRequiresMaterialize"), BooleanValue(fpn.typeRequiresMaterialize))
        builder += typeRequiresMaterialize
      }
      case vdn: VarDirective => {
        val materialize = NameValuePair(NameValue("materialize"), BooleanValue(vdn.materialize))
        builder += materialize
      }
      case fpn: TypePatternNode => {
        val typeRequiresMaterialize = NameValuePair(NameValue("typeRequiresMaterialize"), BooleanValue(fpn.typeRequiresMaterialize))
        builder += typeRequiresMaterialize
      }
      case fpn: BinaryOpNode => {
        val typeRequiresMaterialize = NameValuePair(NameValue("name"), StringValue(fpn.operationName))
        builder += typeRequiresMaterialize
      }
      case fpn: UnaryOpNode => {
        val typeRequiresMaterialize = NameValuePair(NameValue("name"), StringValue(fpn.operationName))
        builder += typeRequiresMaterialize
      }
      case fpn: OverloadedFunctionNode => {
        val typeRequiresMaterialize = NameValuePair(NameValue("cacheable"), BooleanValue(fpn.cacheable))
        builder += typeRequiresMaterialize
      }
      case kvp: KeyValuePairNode => {
        val literalKVP = NameValuePair(NameValue("literal"), BooleanValue(kvp.literal))
        builder += literalKVP
      }
      case _ =>
    }
    if (node.location().startPosition != null) {
      val startPosition: Position = node.location().startPosition
      builder += NameValuePair(NameValue("startPos"), StringValue(s"${startPosition.line},${startPosition.column},${startPosition.index}"))

    }
    if (node.location().endPosition != null) {
      val endPosition: Position = node.location().endPosition
      builder += NameValuePair(NameValue("endPos"), StringValue(s"${endPosition.line},${endPosition.column},${endPosition.index}"))
    }
    Some(AttributesValue(builder.result(), node))
  }

  override def evaluate(implicit ctx: EvaluationContext): T = QualifiedName(node.productPrefix, None)
}

object AstWrapper {
  def apply(node: Any): Value[_] = {
    val v = node match {
      case ln: LiteralValueNode[_] if !FunctionType.accepts(ln.value)(null) => {
        implicit val context = EvaluationContext()
        ln.value match {
          case number if (NumberType.accepts(number)) => StringValue(NumberType.coerce(number).evaluate.toBigDecimal.toString())
          case a                                      => a
        }
      }
      case p: Product => new AstWrapper(p)
      case _          => NullValue
    }
    v
  }
}
