package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.interpreted.node.NameSlot
import org.mule.weave.v2.interpreted.node.VariableReferenceNode
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBinding
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBindings
import org.mule.weave.v2.interpreted.node.structure.header.VariableTable
import org.mule.weave.v2.parser.annotation.MaterializeVariableAnnotation
import org.mule.weave.v2.parser.annotation.StreamingCapableVariableAnnotation
import org.mule.weave.v2.parser.ast.header.directives.ContentType
import org.mule.weave.v2.parser.ast.header.directives.InputDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.ParseException
import org.mule.weave.v2.scope.Reference

import scala.collection.mutable.ArrayBuffer

trait EngineVariableTransformations extends AstTransformation with EngineDirectiveTransformations {

  val _variablesTable: VariableTable

  val _modulesNameTable: VariableTable

  val _externalBindings: ExternalBindings

  val _modules: ArrayBuffer[VariableTable]

  def createNameSlot(name: String): NameSlot = {
    _variablesTable.createVariable(name)
  }

  def createModuleSlot(name: String): NameSlot = {
    _modulesNameTable.createVariable(name)
  }

  def transformVariableReferenceNode(variable: org.mule.weave.v2.parser.ast.variables.VariableReferenceNode): VariableReferenceNode = {
    val resolveVariable = scopeNavigator().resolveVariable(variable.variable)
    if (resolveVariable.isEmpty) {
      throw new ParseException(s"Unable to resolve variable ${variable.variable.name}", variable.location())
    }
    val reference: Reference = resolveVariable.get
    VariableReferenceNode(transformReference(reference))
  }

  def transformReference(reference: Reference): NameSlot = {
    if (reference.isLocalReference) {
      val name = createNameSlot(reference.referencedNode.name)
      //
      val navigator = reference.scope.astNavigator()
      val mayBeInputDirective: Option[InputDirective] = navigator.parentWithType(reference.referencedNode, classOf[InputDirective])
      mayBeInputDirective match {
        case Some(id) =>
          val canStream = id.variable.annotation(classOf[StreamingCapableVariableAnnotation]).exists(_.canStream)
          val needsMaterialize = id.variable.annotation(classOf[MaterializeVariableAnnotation]).exists(_.needMaterialize)
          _externalBindings.addVariable(ExternalBinding(name, id.mime.orElse(id.dataFormat).map((f) => transformFormat(f)), transformOptionSeq(id.options), canStream, needsMaterialize))
        case _ =>
      }
      name
    } else {
      val moduleSlot: NameSlot = createModuleSlot(reference.moduleSource.get.name)
      val slot: Int = moduleSlot.slot
      if (slot < _modules.length)
        _modules(slot).createVariable(reference.referencedNode.name).fqn(moduleSlot)
      else
        throw new RuntimeException(s"Module for ${reference.referencedNode} was not defined.")
    }
  }

  def transformNameSlot(name: NameIdentifier): NameSlot = {
    createNameSlot(name.name)
  }

}
