package org.mule.weave.v2.runtime.core.functions.collections

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.utils.TimeZoneUtils

object ArrayAppendArrayFunctionValue extends BinaryFunctionValue {
  override val L = ArrayType
  override val R = ArrayType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    //Avoid creating new objects when empty
    val leftArray = leftValue.evaluate
    if (leftArray.isEmpty()) {
      rightValue
    } else {
      val rightArray = rightValue.evaluate
      if (rightArray.isEmpty()) {
        leftValue
      } else {
        ArrayValue(leftArray.concat(rightArray), this)
      }
    }
  }
}

object StringAppendStringFunctionValue extends BinaryFunctionValue {
  override val L = StringType
  override val R = StringType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    StringValue(leftValue, rightValue)
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    /**
      * If function call with one of these argument types gets cached here then it is impossible to call
      * LocalDateAppendLocalTimeFunctionValue and similar functions later in a script as both arguments can be
      * coerced to String and will dispatch to this function (See W-17316952).
      */
    def isTemporalArgument(t: Type) =
      t == LocalDateType || t == LocalDateTimeType || t == LocalTimeType

    if (isTemporalArgument(leftArgumentType) || isTemporalArgument(rightArgumentType)) {
      return false
    }

    true
  }
}

object BinaryAppendBinaryFunctionValue extends BinaryFunctionValue {
  override val L = BinaryType
  override val R = BinaryType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    BinaryValue(SeekableStream.append(leftValue.evaluate, rightValue.evaluate))
  }
}

object LocalDateAppendLocalTimeFunctionValue extends BinaryFunctionValue {
  override val L = LocalDateType

  override val R = LocalTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalDateTimeValue(leftValue.evaluate.atTime(rightValue.evaluate), UnknownLocationCapable)
  }
}

object LocalTimeAppendLocalDateFunctionValue extends BinaryFunctionValue {
  override val L = LocalTimeType

  override val R = LocalDateType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalDateAppendLocalTimeFunctionValue.doExecute(rightValue, leftValue)
  }

}

object LocalDateAppendTimeFunctionValue extends BinaryFunctionValue {
  override val L = LocalDateType

  override val R = TimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    DateTimeValue(leftValue.evaluate.atTime(rightValue.evaluate).toZonedDateTime, UnknownLocationCapable)
  }

}

object TimeAppendLocalDateFunctionValue extends BinaryFunctionValue {
  override val L = TimeType

  override val R = LocalDateType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalDateAppendTimeFunctionValue.doExecute(rightValue, leftValue)
  }
}

object LocalDateAppendTimeZoneFunctionValue extends BinaryFunctionValue {
  override val L = LocalDateType

  override val R = TimeZoneType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    DateTimeValue(leftValue.evaluate.atStartOfDay(rightValue.evaluate), UnknownLocationCapable)
  }
}

object TimeZoneAppendLocalDateFunctionValue extends BinaryFunctionValue {
  override val L = TimeZoneType

  override val R = LocalDateType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalDateAppendTimeZoneFunctionValue.doExecute(rightValue, leftValue)
  }
}

object LocalDateTimeAppendTimeZoneFunctionValue extends BinaryFunctionValue {
  override val L = LocalDateTimeType

  override val R = TimeZoneType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    DateTimeValue(leftValue.evaluate.atZone(rightValue.evaluate), UnknownLocationCapable)
  }
}

object TimeZoneAppendLocalDateTimeFunctionValue extends BinaryFunctionValue {
  override val L = TimeZoneType

  override val R = LocalDateTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalDateTimeAppendTimeZoneFunctionValue.doExecute(rightValue, leftValue)
  }
}

object LocalTimeAppendTimeZoneFunctionValue extends BinaryFunctionValue {
  override val L = LocalTimeType

  override val R = TimeZoneType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    TimeValue(leftValue.evaluate.atOffset(TimeZoneUtils.toZoneOffset(rightValue.evaluate)), UnknownLocationCapable)
  }
}

object TimeZoneValueAppendLocalTimeFunctionValue extends BinaryFunctionValue {
  override val L = TimeZoneType

  override val R = LocalTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalTimeAppendTimeZoneFunctionValue.doExecute(rightValue, leftValue)
  }
}

object ObjectAppendObjectFunctionValue extends BinaryFunctionValue {
  override val L = ObjectType
  override val R = ObjectType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    ObjectValue(leftValue.evaluate.append(rightValue.evaluate), UnknownLocationCapable)
  }
}

