package org.mule.weave.v2.runtime.core.functions.collections

import java.util.regex.Matcher

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.types.AnyType
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.RegexType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.math.Number

import scala.collection.mutable.ArrayBuffer

object ArrayFindFunctionValue extends BinaryFunctionValue {

  override val L = ArrayType

  override val R = AnyType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val indexes = leftValue.evaluate
      .toIterator()
      .zipWithIndex
      .filter((pair) => {
        pair._1.equals(rightValue)
      })
      .map((pair) => {
        NumberValue(pair._2)
      })
    ArrayValue(ArraySeq(indexes), this)
  }
}

object StringFindRegexFunctionValue extends BinaryFunctionValue {

  override val L = StringType

  override val R = RegexType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val resultBuilder = new ArrayBuffer[Value[_]]
    val pattern = rightValue.evaluate.pattern
    val matcher: Matcher = ctx.serviceManager.patternService.matcher(pattern, leftValue.evaluate, this)
    while (matcher.find()) {
      val matchBuilder = new ArrayBuffer[NumberValue]
      val count: Int = matcher.groupCount()
      if (count > 0) {
        var index = 0
        while (index <= count) {
          val group: Int = matcher.start(index)
          matchBuilder.+=(NumberValue(Number(group), this))
          index = index + 1
        }
        resultBuilder += ArrayValue(ArraySeq(matchBuilder.result()), this)
      } else {
        resultBuilder += ArrayValue(Seq(NumberValue(Number(matcher.start()))), this)
      }
    }
    ArrayValue(ArraySeq(resultBuilder.result()), this)
  }
}

object StringFindStringFunctionValue extends BinaryFunctionValue {

  override val L = StringType

  override val R = StringType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {

    val str = leftValue.evaluate.toString
    val findStr = rightValue.evaluate.toString

    val findStrLength = Math.max(findStr.length, 1)

    val numberIterator: Iterator[Value[Number]] = new Iterator[Value[Number]]() {
      var value: Int = str.indexOf(findStr, 0)

      override def hasNext: Boolean = {
        value >= 0 && value < str.length
      }

      override def next(): Value[Number] = {
        if (hasNext) {
          val result = NumberValue(value)
          val location = value + findStrLength
          value = str.indexOf(findStr, location)
          result
        } else {
          Iterator.empty.next()
        }
      }
    }

    ArrayValue(ArraySeq(numberIterator), this)
  }
}
