package org.mule.weave.v2.runtime.core.functions.collections

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.structure.function.ExecutionContextAwareFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.MaterializedCollectionMapObjectSeq
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.math.Number

import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

object ArrayGroupByFunctionValue extends BinaryFunctionValue {
  val L = ArrayType

  val R = FunctionType

  def doGroupBy(lhs: L.V, rhs: R.V, inline: Boolean)(implicit ctx: EvaluationContext): mutable.LinkedHashMap[String, ArrayBuffer[Value[_]]] = {
    val result = new mutable.LinkedHashMap[String, ArrayBuffer[Value[_]]]()
    var i = 0
    val elements = lhs.evaluate.toIterator()
    while (elements.hasNext) {
      val value = elements.next().materialize
      val invoke =
        if (inline) {
          rhs.callInline(value, NumberValue(i, this))
        } else {
          rhs.call(value, NumberValue(i, this))
        }
      val key = StringType.coerce(invoke, this).evaluate.toString
      val arrayBuffer = result.getOrElseUpdate(key, ArrayBuffer[Value[_]]())
      arrayBuffer.+=(value)
      i = i + 1
    }
    result
  }

  override def doExecute(lhs: L.V, rhs: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val groupByResult: mutable.Map[String, ArrayBuffer[Value[_]]] =
      rhs match {
        case bf: ExecutionContextAwareFunctionValue if (ctx.isInstanceOf[ExecutionContext]) => {
          val frame = bf.functionContextFrame.child(rhs)
          val context = ctx.asInstanceOf[ExecutionContext]
          context.runInFrame(frame, {
            doGroupBy(lhs, rhs, inline = true)
          })
        }
        case _ => {
          doGroupBy(lhs, rhs, inline = false)
        }
      }

    val stringToValue: collection.Map[String, Value[_]] = groupByResult.mapValues((seq) => ArrayValue(ArraySeq(seq, materialized = true), UnknownLocationCapable))
    ObjectValue(new MaterializedCollectionMapObjectSeq(stringToValue))
  }
}

object ObjectGroupByFunctionValue extends BinaryFunctionValue {
  val L = ObjectType

  val R = FunctionType

  def doGroupBy(lhs: L.V, rhs: R.V, inline: Boolean)(implicit ctx: EvaluationContext): mutable.LinkedHashMap[String, ArrayBuffer[KeyValuePair]] = {
    val groupByResult = new mutable.LinkedHashMap[String, ArrayBuffer[KeyValuePair]]()
    var i = 0
    val elements = lhs.evaluate.toIterator()
    while (elements.hasNext) {
      val keyValuePair = elements.next().materialize
      val indexValue = NumberValue(Number(i), this)
      val invoke = if (inline) {
        rhs.callInline(AttributeDelegateValue(keyValuePair), keyValuePair._1, indexValue)
      } else {
        rhs.call(AttributeDelegateValue(keyValuePair), keyValuePair._1, indexValue)
      }
      val key = StringType.coerce(invoke, this).evaluate.toString
      val arrayBuffer = groupByResult.getOrElseUpdate(key, ArrayBuffer[KeyValuePair]())
      arrayBuffer.+=(keyValuePair)
      i = i + 1
    }
    groupByResult
  }

  override def doExecute(lhs: L.V, rhs: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val groupByResult =
      rhs match {
        case bf: ExecutionContextAwareFunctionValue if (ctx.isInstanceOf[ExecutionContext]) => {
          val frame = bf.functionContextFrame.child(rhs)
          val context = ctx.asInstanceOf[ExecutionContext]
          context.runInFrame(frame, {
            doGroupBy(lhs, rhs, inline = true)
          })
        }
        case _ => {
          doGroupBy(lhs, rhs, inline = false)
        }
      }
    val stringToValue: collection.Map[String, Value[_]] = groupByResult.mapValues((keyValuePairs) => ObjectValue(ObjectSeq(keyValuePairs, materialized = true), UnknownLocationCapable))
    ObjectValue(new MaterializedCollectionMapObjectSeq(stringToValue))
  }

}