package org.mule.weave.v2.runtime.core.functions.collections

import org.mule.weave.v2.api.tooling.message.Message
import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.service.MessageLoggingService
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.runtime.core.exception.InvalidComparisonException

object ArrayOrderByFunctionValue extends BinaryFunctionValue {
  val L = ArrayType

  val R = FunctionType

  override def doExecute(lhs: L.V, rhs: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val result = lhs.materialize.evaluate
      .toSeq()
      .zipWithIndex //
      .sortWith((v1: (Value[_], Int), v2: (Value[_], Int)) => {
        val value1 = rhs.call(v1._1, NumberValue(Number(v1._2), this))
        val value2 = rhs.call(v2._1, NumberValue(Number(v2._2), this))
        ValueTypeChecker.checkTypes(value1.valueType.baseType, value2.valueType.baseType, location())
        value1.compareTo(value2) < 0
      })
      .map(_._1)
    ArrayValue(ArraySeq(result), this)
  }
}

object ObjectOrderByFunctionValue extends BinaryFunctionValue {

  val L = ObjectType

  val R = FunctionType

  override def doExecute(lhs: L.V, rhs: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val sorted = lhs.materialize.evaluate
      .toSeq() //
      .sortWith((property1: KeyValuePair, property2: KeyValuePair) => {
        val value1 = rhs.call(AttributeDelegateValue(property1._2, property1._1), property1._1)
        val value2 = rhs.call(AttributeDelegateValue(property2._2, property2._1), property2._1)
        ValueTypeChecker.checkTypes(value1.valueType.baseType, value2.valueType.baseType, location())
        value1.compareTo(value2) < 0
      })
    ObjectValue(ObjectSeq(sorted), this)
  }
}

object ValueTypeChecker {
  def checkTypes(baseType1: Type, baseType2: Type, location: Location)(implicit ctx: EvaluationContext): Unit = {
    if (!baseType1.eq(baseType2)) {
      if (ctx.serviceManager.settingsService.execution().stableOrderBy) {
        throw new InvalidComparisonException(baseType1, baseType2, location)
      } else {
        ctx.serviceManager.messageLoggingService.logWarnDebounce(MixedTypesComparison(baseType1, baseType2))
      }
    }
  }
}

case class MixedTypesComparison(type1: Type, type2: Type) extends Message {

  override def getKind: String = "MixedTypesComparison"

  override def getMessage: String = {
    s"Sorting using values of different types may produce unexpected results (type1: ${type1.name}, type2: ${type2.name})."
  }
}
