package org.mule.weave.v2.runtime.core.operator.math

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.math
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.runtime.core.exception.SubtractionExecutionException
import org.mule.weave.v2.runtime.core.operator.math.exception.InvalidTemporalSubtraction

import java.time.DateTimeException
import java.time.Duration
import java.time.LocalDate
import java.time.LocalTime
import java.time.Period

class NumberSubtractionNumberOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = NumberType

  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val leftNumber = leftValue.evaluate
    val rightNumber = rightValue.evaluate
    NumberValue(doSubtraction(leftNumber, rightNumber), this)
  }

  def doSubtraction(leftNumber: math.Number, rightNumber: math.Number) = {
    try {
      leftNumber - rightNumber
    } catch {
      case ae: ArithmeticException => {
        throw new SubtractionExecutionException(leftNumber.toString, rightNumber.toString, ae.getMessage, location)
      }
    }
  }
}

class ArraySubtractionOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ArrayType

  override val R = AnyType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val right: Value[_] = rightValue.materialize
    ArrayValue(
      leftValue.evaluate
        .toIterator()
        .filterNot((v: Value[_]) => {
          v.equals(right)
        }),
      this)
  }
}

class ObjectSubtractionKeyOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ObjectType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    ObjectValue(leftValue.evaluate.removeKey(rightValue.evaluate), this)
  }
}

class ObjectSubtractionStringOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ObjectType

  override val R = StringType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    ObjectValue(leftValue.evaluate.removeKey(QualifiedName(rightValue.evaluate.toString, None)))
  }
}

class LocalDateSubtractionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = LocalDateType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalDateValue(leftValue.evaluate.minus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

object LocalDateSubtractionPeriodOperator extends LocalDateSubtractionPeriodOperator(UnknownLocation)

class PeriodSubtractionLocalDateOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = PeriodType

  override val R = LocalDateType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalDateSubtractionPeriodOperator.doExecute(rightValue, leftValue)
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

class LocalDateSubtractLocalDateOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = LocalDateType

  override val R = LocalDateType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    if (ctx.serviceManager.settingsService.execution().dateMinusCompatibility) {
      PeriodValue(Period.between(rightValue.evaluate, leftValue.evaluate), this)
    } else {
      val rightDate: LocalDate = rightValue.evaluate
      val leftDate: LocalDate = leftValue.evaluate
      PeriodValue(Duration.between(rightDate.atTime(LocalTime.MIN), leftDate.atTime(LocalTime.MIN)), this)
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allowLeft = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }

    val allow = if (allowLeft) {
      rightArgumentType match {
        case _: StringType => false
        case _             => true
      }
    } else {
      allowLeft
    }
    allow
  }
}

class LocalDateTimeSubtractionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = LocalDateTimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalDateTimeValue(leftValue.evaluate.minus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }

}

object LocalDateTimeSubtractionPeriodOperator extends LocalDateTimeSubtractionPeriodOperator(UnknownLocation)

class PeriodSubtractionLocalDateTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = PeriodType

  override val R = LocalDateTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalDateTimeSubtractionPeriodOperator.doExecute(rightValue, leftValue)
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }

}

class LocalDateTimeSubtractLocalDateTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = LocalDateTimeType

  override val R = LocalDateTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      PeriodValue(Duration.between(rightValue.evaluate, leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allowLeft = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    val allow = if (allowLeft) {
      rightArgumentType match {
        case _: StringType => false
        case _             => true
      }
    } else {
      allowLeft
    }
    allow
  }
}

class DateTimeSubtractionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = DateTimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      DateTimeValue(leftValue.evaluate.minus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

object DateTimeSubtractionPeriodOperator extends DateTimeSubtractionPeriodOperator(UnknownLocation)

class PeriodSubtractionDateTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = PeriodType

  override val R = DateTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    DateTimeSubtractionPeriodOperator.doExecute(rightValue, leftValue)
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

class DateTimeSubtractDateTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = DateTimeType

  override val R = DateTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      PeriodValue(Duration.between(rightValue.evaluate, leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allowLeft = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    val allow = if (allowLeft) {
      rightArgumentType match {
        case _: StringType => false
        case _             => true
      }
    } else {
      allowLeft
    }
    allow
  }
}

class TimeSubtractionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = TimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      TimeValue(leftValue.evaluate.minus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }
}

object TimeSubtractionPeriodOperator extends TimeSubtractionPeriodOperator(UnknownLocation)

class PeriodSubtractionTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = PeriodType

  override val R = TimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    TimeSubtractionPeriodOperator.doExecute(rightValue, leftValue)
  }
}

class TimeSubtractTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = TimeType

  override val R = TimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      PeriodValue(Duration.between(rightValue.evaluate, leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }
}

class LocalTimeSubtractionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = LocalTimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalTimeValue(leftValue.evaluate.minus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }

}

object LocalTimeSubtractionPeriodOperator extends LocalTimeSubtractionPeriodOperator(UnknownLocation)

class PeriodSubtractionLocalTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = PeriodType

  override val R = LocalTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    LocalTimeSubtractionPeriodOperator.doExecute(rightValue, leftValue)
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

class LocalTimeSubtractLocalTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = LocalTimeType

  override val R = LocalTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      PeriodValue(Duration.between(rightValue.evaluate, leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalSubtraction(e.getMessage, location)
      }
    }
  }
}
