package org.mule.weave.v2.runtime.core.operator.selectors

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.core.exception
import org.mule.weave.v2.core.exception.InvalidSelectionException
import org.mule.weave.v2.core.exception.StrictSelectionOverNullSelection
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.AttributeDelegateValue
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation

import scala.util.Try

class ArrayIndexSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = ArrayType

  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val arr = leftValue.evaluate
    val index: Long = rightValue.evaluate.toLong
    if (index >= 0) {
      arr.apply(index) match {
        case Some(value) => value
        case None        => throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, arr.size()))
      }
    } else {
      //We need the seq so that we can calculate the size and get it by index
      val array = arr.toArray()
      try {
        array.apply((array.length + index).toInt)
      } catch {
        case _: Exception => throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, array.length))
      }
    }
  }
}

class ObjectIndexSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = ObjectType

  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val objectSeq = leftValue.evaluate
    val index: Int = rightValue.evaluate.toInt
    if (index >= 0) {
      objectSeq.apply(index) match {
        case value: KeyValuePair => AttributeDelegateValue(value)
        case _                   => throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, objectSeq.size))
      }
    } else {
      //We need the seq so that we can calculate the size and get it by index
      val toSeq = objectSeq.toSeq()
      Try(toSeq.apply(toSeq.size + index))
        .map(AttributeDelegateValue(_))
        .getOrElse(throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, toSeq.size)))
    }
  }
}

class StringIndexSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = StringType

  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val arr = leftValue.evaluate
    val index: Int = rightValue.evaluate.toInt
    try {
      if (index >= 0) {
        StringValue(arr.charAt(index).toString)
      } else {
        StringValue(arr.charAt(arr.length + index).toString)
      }
    } catch {
      case _: java.lang.IndexOutOfBoundsException => throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, arr.length))
    }
  }
}

class BinaryIndexSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = BinaryType

  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val binary = leftValue.evaluate
    val index: Long = rightValue.evaluate.toLong
    try {
      if (index >= 0) {
        binary.seek(index)
        BinaryValue(Array(binary.read().toByte))
      } else {
        binary.seek(binary.size() + index)
        BinaryValue(Array(binary.read().toByte))
      }
    } catch {
      case _: java.lang.IndexOutOfBoundsException => throw InvalidSelectionException(new exception.IndexOutOfBoundsException(this.location, index, binary.size()))
    }
  }
}

class NullIndexSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = NullType

  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    throw InvalidSelectionException(new StrictSelectionOverNullSelection(this))
  }
}