package org.mule.weave.v2.interpreted.module.reader

import org.mule.weave.v2.core.util.BinaryHelper
import org.mule.weave.v2.core.exception.InvalidNumberExpressionException
import org.mule.weave.v2.grammar.literals.DateLiteral
import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.interpreted.module.reader.exception.WeaveReaderException
import org.mule.weave.v2.interpreted.transform.DateHelper
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.NameValuePair
import org.mule.weave.v2.model.structure.Namespace
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.values.AlreadyMaterializedObjectValue
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.AttributesValue
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.BooleanValue
import org.mule.weave.v2.model.values.DateTimeValue
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.LocalDateTimeValue
import org.mule.weave.v2.model.values.LocalDateValue
import org.mule.weave.v2.model.values.LocalTimeValue
import org.mule.weave.v2.model.values.NameValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.PeriodValue
import org.mule.weave.v2.model.values.RegexValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.TimeValue
import org.mule.weave.v2.model.values.TimeZoneValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.model.values.wrappers.AsValue
import org.mule.weave.v2.module.reader.ReaderInput
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.reader.SourceReader
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.ast.structure.DateTimeNode
import org.mule.weave.v2.parser.ast.structure.LocalDateNode
import org.mule.weave.v2.parser.ast.structure.LocalDateTimeNode
import org.mule.weave.v2.parser.ast.structure.LocalTimeNode
import org.mule.weave.v2.parser.ast.structure.PeriodNode
import org.mule.weave.v2.parser.ast.structure.TimeNode
import org.mule.weave.v2.parser.ast.structure.TimeZoneNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.DefaultLocationCapable
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.utils.WeaveConstants
import org.parboiled2.ErrorFormatter
import org.parboiled2.ParseError
import org.parboiled2.Parser
import org.parboiled2.ParserInput

import java.lang
import java.lang.{ StringBuilder => JStringBuilder }
import java.time.Duration
import java.time.LocalDateTime
import java.time.LocalTime
import java.time.OffsetTime
import java.time.Period
import java.time.ZoneId
import java.time.ZoneOffset
import java.time.ZonedDateTime
import scala.annotation.switch
import scala.annotation.tailrec
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.util.Failure
import scala.util.Success
import scala.util.Try

/**
  * Weave data format parser. It only parses weave data structures of literal values.
  */
class OnlyDataInMemoryWeaveParser(override val name: String, val sourceProvider: SourceProvider)(implicit ctx: EvaluationContext) extends WeaveLiteralTokenizer {

  def parse(): Value[_] = {
    advance()
    retrieveNextValue(new NamespaceContext())
  }

  def retrieveRegex(): Value[_] = {
    val strBuilder = new JStringBuilder()
    var read = true
    advance()
    while (read) {
      cursorChar match {
        case '\\' => {
          advance()
          if (advanceIf('/')) {
            strBuilder.append('/')
          } else {
            strBuilder.append('\\')
          }
        }
        case ReaderInput.EOI => {
          fail("End of input reached but expected '/'")
        }
        case '/' => {
          advance()
          read = false
        }
        case _ => {
          strBuilder.append(cursorChar)
          advance()
        }
      }
    }
    ws()
    RegexValue(strBuilder.toString)
  }

  def retrieveTemporal(): Value[_] = {
    val strBuilder = new JStringBuilder()
    val startLocation = location()
    var read = true
    advance()
    while (read) {
      cursorChar match {
        case '|' => {
          advance()
          read = false
        }
        case ReaderInput.EOI => {
          fail("End of input reached but expected '|'")
        }
        case _ => {
          strBuilder.append(cursorChar)
          advance()
        }
      }
    }
    ws()
    val input = new SafeStringBasedParserInput(strBuilder.toString)
    val value = new DateParser(input).parse()
    value match {
      case Failure(y: ParseError) => {
        val format: String = y.format(input, new ErrorFormatter() {
          override def formatErrorLine(sb: lang.StringBuilder, error: ParseError, input: ParserInput): lang.StringBuilder = {
            sb
          }
        })
        throw new WeaveReaderException("Problem while parsing date : " + format, startLocation)
      }
      case Failure(error) => {
        throw new WeaveReaderException("Problem while parsing date : " + error.getMessage, startLocation)
      }
      case Success(dateValue) => {
        dateValue
      }
    }
  }

  def readUntilWS(): String = {
    val result = new JStringBuilder()
    var read = true
    while (read) {
      cursorChar match {
        case ' ' | '\t' | '\n' => read = false
        case c => {
          result.append(c)
          advance()
        }
      }
    }
    ws()
    result.toString
  }

  def retrieveDoBlock(context: NamespaceContext): Value[_] = {
    context.push()
    require('d')
    require('o')
    ws()
    require('{')
    ws()
    var read = true
    while (read)
      cursorChar match {
        case 'n' => {
          require('n')
          require('s')
          ws()
          val prefix = readUntilWS()
          val uri = readUntilWS()
          context.add(prefix, uri)
        }
        case '-' => {
          read = false
        }
      }
    require('-')
    require('-')
    require('-')
    ws()
    val result = retrieveNextValue(context)
    require('}')
    ws()

    context.pop()
    result
  }

  def retrieveSchema(context: NamespaceContext): Schema = {
    val schemaProperties = new ArrayBuffer[SchemaProperty]()
    require('{')
    ws()
    if (cursorChar != '}')
      readSchemaProperties(context, schemaProperties)
    require('}')
    ws()
    Schema(schemaProperties)
  }

  @tailrec
  private def readSchemaProperties(context: NamespaceContext, kvps: ArrayBuffer[SchemaProperty]): Unit = {
    val key = readQName(context).name
    require(':')
    ws()
    val value = retrieveNextValue(context)
    kvps += SchemaProperty(StringValue(key), value)
    if (ws(',')) {
      readSchemaProperties(context, kvps)
    }
  }

  def retrieveNextValue(context: NamespaceContext): Value[_] = {
    ws()
    val value = (cursorChar: @switch) match {
      case 'f' => {
        requireFalse()
        BooleanValue.FALSE_BOOL
      }
      case 'd' => {
        retrieveDoBlock(context)
      }
      case 't' => {
        requireTrue()
        BooleanValue.TRUE_BOOL
      }
      case 'n' => {
        requireNull()
        NullValue
      }
      case '/' => retrieveRegex()
      case '|' => retrieveTemporal()
      case '{' => retrieveObject(context)
      case '[' => retrieveArray(context)
      case '"' => {
        val str = readString()
        ws()
        StringValue(str)
      }
      case '0' | '1' | '2' | '3' | '4' | '5' | '6' | '7' | '8' | '9' | '-' => {
        val number = readNumber()
        ws()
        NumberValue(number)
      }
      case _ => {
        fail(s"false or true or null or {...} or [...] or number or /regex/ or |date| but was `${cursorChar}`.", location())
      }
    }

    cursorChar match {
      case 'a' =>
        require('a')
        require('s')
        ws()
        val name = readUnquotedString()
        ws()
        val valueType = value.valueType
        val schema = retrieveSchema(context)
        //This is a hack but is the way weave writer persists the binary values
        //So in order to have back a binary value we should keep it
        if (name.equals(TypeLiteral.BINARY_TYPE_NAME) && schema.base.isDefined && value.isInstanceOf[StringValue]) {
          val bytes = BinaryHelper.fromBase64String(value.asInstanceOf[StringValue].evaluate, UnknownLocationCapable)
          BinaryValue(bytes)
        } else {
          new AsValue(value, Some(schema), value)
        }
      case _ => value
    }
  }

  def readString(): String = {
    require('"')
    val str = parseString()
    require('"')
    str
  }

  private def readNumber(): Number = {
    val builder = new JStringBuilder()
    ch('-', builder)
    readInt(builder)
    readFrac(builder)
    readExp(builder)
    val numberStr = builder.toString
    try {
      Number(numberStr)
    } catch {
      case _: InvalidNumberExpressionException => {
        throw new WeaveReaderException(s"Invalid number format ${numberStr}", location())
      }
    }
  }

  private def readFrac(builder: JStringBuilder): Unit = {
    if (ch('.', builder)) readOneOrMoreDigits(builder)
  }

  private def readExp(builder: JStringBuilder): Unit = if (ch('e', builder) || ch('E', builder)) {
    ch('-', builder) || ch('+', builder)
    readOneOrMoreDigits(builder)
  }

  private def readInt(builder: JStringBuilder): Unit = {
    if (!ch('0', builder)) readOneOrMoreDigits(builder)
  }

  private def ch(c: Char, builder: JStringBuilder): Boolean = {
    val ret = advanceIf(c)
    if (ret) builder.append(c)
    ret
  }

  private def readOneOrMoreDigits(builder: JStringBuilder): Unit = {
    if (readDigit(builder)) {
      readZeroOrMoreDigits(builder)
    } else {
      fail("readDigit")
    }
  }

  @tailrec protected final def readZeroOrMoreDigits(builder: JStringBuilder): Unit = {
    if (readDigit(builder)) readZeroOrMoreDigits(builder)
  }

  protected def readDigit(builder: JStringBuilder): Boolean = {
    isDigit && advance(builder)
  }

  private def isDigit = {
    cursorChar >= '0' && cursorChar <= '9'
  }

  private def advance(builder: JStringBuilder): Boolean = {
    builder.append(cursorChar)
    advance()
  }

  private def retrieveArray(context: NamespaceContext): ArrayValue = {
    val values = new ArrayBuffer[Value[_]]
    advance()
    ws()
    if (cursorChar != ']') {
      readArrayMembers(context, values)
    }
    require(']')
    ws()
    val readerLocation = location()
    ArrayValue(ArraySeq(values, materialized = true), DefaultLocationCapable(readerLocation))
  }

  @tailrec
  private def readArrayMembers(context: NamespaceContext, values: ArrayBuffer[Value[_]]): Unit = {
    val value = retrieveNextValue(context)
    values += value
    if (ws(',')) {
      readArrayMembers(context, values)
    }
  }

  private def retrieveObject(context: NamespaceContext): ObjectValue = {
    val kvps = new ArrayBuffer[KeyValuePair]
    advance()
    ws()
    if (cursorChar != '}') {
      readObjectMembers(context, kvps)
    }
    ws()
    require('}')
    ws()
    val readerLocation = location()
    ObjectValue(ObjectSeq(kvps, materialized = true), DefaultLocationCapable(readerLocation))
  }

  @tailrec
  private def readObjectMembers(context: NamespaceContext, kvps: ArrayBuffer[KeyValuePair]): Unit = {
    val key = readKey(context)
    require(':')
    ws()
    val value = retrieveNextValue(context)
    kvps += KeyValuePair(key, value)
    if (ws(',')) {
      readObjectMembers(context, kvps)
    }
  }

  @tailrec
  private def readAttributeMembers(context: NamespaceContext, kvps: ArrayBuffer[NameValuePair]): Unit = {
    val key = readQName(context)
    require(':')
    ws()
    val value = retrieveNextValue(context)
    kvps += NameValuePair(NameValue(key), value)
    if (ws(',')) {
      readAttributeMembers(context, kvps)
    }
  }

  def readUnquotedString(): String = {
    val result = new JStringBuilder()
    var read = true
    while (read) {
      cursorChar match {
        case ' ' | '#' | '\t' | '\n' | ':' =>
          read = false
        case _ => {
          result.append(cursorChar)
          advance()
        }
      }
    }
    result.toString
  }

  private def readKey(context: NamespaceContext): KeyValue = {
    ws()
    val qname: QualifiedName = readQName(context)
    val attributes = cursorChar match {
      case '@' => {
        advance()
        require('(')
        ws()
        val pairs = new ArrayBuffer[NameValuePair]()
        if (cursorChar != ')') {
          readAttributeMembers(context, pairs)
        }
        ws()
        require(')')
        Some(AttributesValue(pairs))
      }
      case _ => None
    }
    ws()
    KeyValue(qname, attributes)
  }

  private def readQName(context: NamespaceContext): QualifiedName = {
    val qname = cursorChar match {
      case '"' => {
        QualifiedName(readString(), None)
      }
      case _ =>
        val firstPart = readUnquotedString()
        ws()
        if (advanceCharIf('#')) {
          val name = cursorChar match {
            case '"' => readString()
            case _   => readUnquotedString()
          }
          val maybeString = context.resolve(firstPart)
          if (maybeString.isEmpty) {
            fail(s"Unable to resolve ns ${firstPart}")
          }
          QualifiedName(name, Some(Namespace(firstPart, maybeString.get)))
        } else {
          QualifiedName(firstPart, None)
        }
    }
    ws()
    qname
  }

  def appendCharAndAdvance(builder: JStringBuilder): Boolean = {
    builder.append(cursorChar)
    advance()
  }

  def canReadChar: Boolean = {
    ((1L << cursorChar) & ((31 - cursorChar) >> 31) & 0x7ffffffbefffffffL) != 0L
  }

  override val input: SourceReader = {
    SourceReader(sourceProvider, Some(WeaveConstants.default_charset))
  }
}

class InMemoryObject(objectSeq: ObjectSeq, val location: Location) extends ObjectValue with AlreadyMaterializedObjectValue {

  override def evaluate(implicit ctx: EvaluationContext): T = objectSeq

}

class DateParser(val input: SafeStringBasedParserInput) extends Parser with DateLiteral {

  override def resourceName: NameIdentifier = NameIdentifier.anonymous

  def parse(): Try[Value[_]] = {
    val triedNode = anyDateExpression.run()
    triedNode.flatMap((value) => {
      Try(
        value match {
          case LocalTimeNode(time, _) => LocalTimeValue(LocalTime.parse(time))
          case TimeNode(time, _)      => TimeValue(OffsetTime.parse(time))
          case TimeZoneNode(zoneStr, _) => {
            val id = Try(ZoneOffset.of(zoneStr))
              .getOrElse(ZoneId.of(zoneStr, ZoneId.SHORT_IDS))
            TimeZoneValue(id)
          }
          case LocalDateNode(localdateStr, _) => {
            val date = DateHelper.parseLocalDate(localdateStr)
            LocalDateValue(date)
          }
          case PeriodNode(period, _) => {
            if (period.contains('T')) {
              PeriodValue(Duration.parse(period))
            } else {
              PeriodValue(Period.parse(period))
            }
          }
          case LocalDateTimeNode(dateTime, _) => {
            LocalDateTimeValue(LocalDateTime.parse(dateTime))
          }
          case DateTimeNode(dateTime, _) => {
            DateTimeValue(ZonedDateTime.parse(dateTime))
          }
        })
    })
  }

  override def attachDocumentation: Boolean = false
}

class NamespaceContext {
  val context: ArrayBuffer[mutable.Map[String, String]] = ArrayBuffer()

  def push(): Unit = {
    context.+=(new mutable.HashMap[String, String]())
  }

  def resolve(prefix: String): Option[String] = {
    var result: Option[String] = None
    var i = context.length - 1
    while (i >= 0 && result.isEmpty) {
      result = context.apply(i).get(prefix)
      i = i - 1
    }
    result
  }

  def add(prefix: String, uri: String): Option[String] = {
    context.last.put(prefix, uri)
  }

  def pop(): mutable.Map[String, String] = {
    context.remove(context.size - 1)
  }
}