package org.mule.weave.v2.interpreted.node.structure

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.structure.NameSeq
import org.mule.weave.v2.model.structure.Namespace
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.Value

class LiteralNameKeyNode(val keyName: StringNode, val ns: Option[ValueNode[Namespace]], val attr: Option[ValueNode[NameSeq]]) extends ValueNode[QualifiedName] {

  var name: QualifiedName = _
  var staticKey: KeyValue = _

  override def doExecute(implicit executionContext: ExecutionContext): Value[QualifiedName] = {
    if (name == null) {
      val key: String = keyName.execute.evaluate
      val namespace: Option[Namespace] = ns.map(_.execute.evaluate)
      name = QualifiedName(key, namespace)
    }
    val attributes: Option[Value[NameSeq]] = attr.map(_.execute)
    KeyValue(name, attributes, this)
  }

  override def productElement(n: Int): Any = {
    n match {
      case 0 => keyName
      case 1 => attr.orElse(ns).orNull
      case 2 => ns.orNull
    }
  }

  override def productArity: Int = {
    var arity = 1
    if (attr.isDefined) arity = arity + 1
    if (ns.isDefined) arity = arity + 1
    arity
  }
}

class LiteralKeyNode(val keyName: StringNode, val ns: Option[ValueNode[Namespace]], val attr: Option[ValueNode[NameSeq]]) extends ValueNode[QualifiedName] {

  var staticKey: KeyValue = _

  override def doExecute(implicit executionContext: ExecutionContext): Value[QualifiedName] = {
    if (staticKey == null) {
      val key: String = keyName.execute.evaluate
      val namespace: Option[Namespace] = ns.map(_.execute.evaluate)
      val name = QualifiedName(key, namespace)
      val attributes: Option[Value[NameSeq]] = attr.map(_.execute)
      staticKey = KeyValue(name, attributes, this)
    }
    staticKey
  }

  override def productElement(n: Int): Any = {
    n match {
      case 0 => keyName
      case 1 => attr.orElse(ns).orNull
      case 2 => ns.orNull
    }
  }

  override def productArity: Int = {
    var arity = 1
    if (attr.isDefined) arity = arity + 1
    if (ns.isDefined) arity = arity + 1
    arity
  }

  override def shouldNotify: Boolean = false
}

class KeyNode(val keyName: ValueNode[StringType#T], val ns: Option[ValueNode[Namespace]], val attr: Option[ValueNode[NameSeq]]) extends ValueNode[QualifiedName] {
  override def doExecute(implicit executionContext: ExecutionContext): Value[QualifiedName] = {
    val key: StringType#T = keyName.execute.evaluate
    val namespace: Option[Namespace] = ns.map(_.execute.evaluate)
    val attributes: Option[Value[NameSeq]] = attr.map(_.execute)
    KeyValue(QualifiedName(key.toString, namespace), attributes, this)
  }

  override def productElement(n: Int): Any = {
    n match {
      case 0 => keyName
      case 1 => attr.orElse(ns).orNull
      case 2 => ns.orNull
    }
  }

  override def productArity: Int = {
    var arity = 1
    if (attr.isDefined) arity = arity + 1
    if (ns.isDefined) arity = arity + 1
    arity
  }
}

class DynamicKeyNode(val keyExpression: ValueNode[_], val attr: Option[ValueNode[NameSeq]]) extends ValueNode[QualifiedName] {
  override protected def doExecute(implicit ctx: ExecutionContext): Value[QualifiedName] = {
    val executionResult = keyExpression.execute
    val key = executionResult match {
      case aKey if KeyType.accepts(executionResult) => aKey.asInstanceOf[KeyType.V]
      case aStr if StringType.accepts(executionResult) => KeyValue(aStr.evaluate.asInstanceOf[StringType.T].toString)
      case _ => KeyType.coerce(executionResult, this)
    }

    attr match {
      case Some(attrs) => KeyValue(key.evaluate, Some(attrs.execute), this)
      case None        => key
    }
  }

  override def productElement(n: Int): Any = {
    n match {
      case 0 => keyExpression
      case 1 => attr.orNull
    }
  }

  override def productArity: Int = {
    var arity = 1
    if (attr.isDefined) arity = arity + 1
    arity
  }
}

object KeyNode {
  def apply(keyName: ValueNode[StringType#T], ns: Option[ValueNode[Namespace]], attr: Option[ValueNode[NameSeq]]): ValueNode[QualifiedName] = {
    new KeyNode(keyName, ns, attr)
  }

  def literalKeyNode(keyName: ValueNode[StringType#T], ns: Option[ValueNode[Namespace]], attr: Option[ValueNode[NameSeq]]): ValueNode[QualifiedName] = {
    keyName match {
      case strNode: StringNode => new LiteralKeyNode(strNode, ns, attr)
      case _                   => new KeyNode(keyName, ns, attr)
    }
  }

  def literalNameKeyNode(keyName: ValueNode[StringType#T], ns: Option[ValueNode[Namespace]], attr: Option[ValueNode[NameSeq]]): ValueNode[QualifiedName] = {
    keyName match {
      case strNode: StringNode => new LiteralNameKeyNode(strNode, ns, attr)
      case _                   => new KeyNode(keyName, ns, attr)
    }
  }

  def apply(keyExpression: ValueNode[_], attr: Option[ValueNode[NameSeq]]) = new DynamicKeyNode(keyExpression, attr)
}
