package org.mule.weave.v2.interpreted

import org.mule.weave.v2.interpreted.node.ModuleNode
import org.mule.weave.v2.interpreted.transform.EngineGrammarTransformation
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.utils.LockFactory

import java.util.concurrent.ConcurrentHashMap

/**
  * Handles the transformation to a Executable ModuleNode.
  * It goes from an fully qualified name, look it up and executes the corresponding compilation phases.
  */
trait RuntimeModuleNodeCompiler {
  /**
    * Compiles the module with the specified NameIdentifier
    *
    * @param moduleName     The name of the module to be compiled
    * @param parsingContext The parsing context
    * @return The runtime module node if it was able to successfully compile it.
    */
  def compile(moduleName: NameIdentifier, parsingContext: ParsingContext): Option[ModuleNode]

  /**
    * Compiles the module with the specified NameIdentifier
    *
    * @param moduleName     The name of the module to be compiled
    * @param parsingContext The parsing context
    * @param rootCompiler   The root compiler in the chain of compilers.
    * @return The runtime module node if it was able to successfully compile it.
    */
  def compile(moduleName: NameIdentifier, parsingContext: ParsingContext, rootCompiler: RuntimeModuleNodeCompiler): Option[ModuleNode]

  /**
    * Invalidate a cached entry with the given NameIdentifier
    *
    * @param nameIdentifier The name identifier of the module to invalidate.
    */
  def invalidate(nameIdentifier: NameIdentifier): Unit
}

/**
  * A runtime module compiler
  *
  * @param parserManager If a parser manager is specified then it will use this one to look for resources. Else it will use the one in the parsing context
  */
class DefaultRuntimeModuleNodeCompiler(parserManager: Option[ModuleParsingPhasesManager]) extends RuntimeModuleNodeCompiler {

  private val modules = new ConcurrentHashMap[NameIdentifier, ModuleNode]()
  private val modulesLock = LockFactory.createLock()

  def this() = {
    this(None)
  }

  override def compile(moduleName: NameIdentifier, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): Option[ModuleNode] = {
    parsingContext.notificationManager.startCompilation(moduleName)
    val maybeModule = modules.get(moduleName)
    if (maybeModule != null) {
      parsingContext.notificationManager.endCompilation(moduleName, cached = true, Some(maybeModule))
      Some(maybeModule)
    } else {
      modulesLock.lock(
        moduleName, {
        val maybeModule = modules.get(moduleName)
        if (maybeModule != null) {
          Some(maybeModule)
        } else {
          val parsingPhasesManager = parserManager.getOrElse(parsingContext.moduleParserManager)
          val maybeResult = parsingPhasesManager.scopeCheckModule(moduleName, parsingContext)
          maybeResult.map(result => {
            val moduleNode = EngineGrammarTransformation(parsingContext.child(moduleName), result.getResult().scope, moduleNodeLoader).transformModule(result.getResult().astNode)
            modules.put(moduleName, moduleNode)
            parsingContext.notificationManager.endCompilation(moduleName, cached = false, Some(moduleNode))
            moduleNode
          })
        }
      })
    }
  }

  def compile(moduleName: NameIdentifier, parsingContext: ParsingContext): Option[ModuleNode] = {
    compile(moduleName, parsingContext, this)
  }

  override def invalidate(nameIdentifier: NameIdentifier): Unit = {
    modules.remove(nameIdentifier)
  }
}

class CompositeRuntimeModuleNodeCompiler(parent: RuntimeModuleNodeCompiler, child: RuntimeModuleNodeCompiler) extends RuntimeModuleNodeCompiler {

  override def compile(moduleName: NameIdentifier, parsingContext: ParsingContext): Option[ModuleNode] = {
    compile(moduleName, parsingContext, this)
  }

  override def compile(moduleName: NameIdentifier, parsingContext: ParsingContext, rootCompiler: RuntimeModuleNodeCompiler): Option[ModuleNode] = {
    parent.compile(moduleName, parsingContext, rootCompiler).orElse(child.compile(moduleName, parsingContext, rootCompiler))
  }

  override def invalidate(nameIdentifier: NameIdentifier): Unit = {
    parent.invalidate(nameIdentifier)
    child.invalidate(nameIdentifier)
  }

}

object CompositeRuntimeModuleNodeCompiler {
  def apply(parent: RuntimeModuleNodeCompiler, child: RuntimeModuleNodeCompiler): CompositeRuntimeModuleNodeCompiler = {
    new CompositeRuntimeModuleNodeCompiler(parent, child)
  }
}

object RuntimeModuleNodeCompiler {

  def apply(): RuntimeModuleNodeCompiler = new DefaultRuntimeModuleNodeCompiler()

  def apply(parserManager: ModuleParsingPhasesManager, parent: Option[RuntimeModuleNodeCompiler] = None): RuntimeModuleNodeCompiler = {
    val compiler = new DefaultRuntimeModuleNodeCompiler(Some(parserManager))
    parent match {
      case Some(parent) => {
        new CompositeRuntimeModuleNodeCompiler(parent, compiler)
      }
      case _ => {
        compiler
      }
    }
  }

  def parentFirst(parserManager: ModuleParsingPhasesManager, parent: RuntimeModuleNodeCompiler): RuntimeModuleNodeCompiler = {
    RuntimeModuleNodeCompiler(parserManager = parserManager, parent = Some(parent))
  }

  def parentLast(parserManager: ModuleParsingPhasesManager, parent: RuntimeModuleNodeCompiler): RuntimeModuleNodeCompiler = {
    val compiler = new DefaultRuntimeModuleNodeCompiler(Some(parserManager))
    new CompositeRuntimeModuleNodeCompiler(compiler, parent)
  }

  def chain(parserManager: ModuleParsingPhasesManager, parent: RuntimeModuleNodeCompiler, parentLast: Boolean): RuntimeModuleNodeCompiler = {
    if (parentLast) {
      RuntimeModuleNodeCompiler.parentLast(parserManager, parent)
    } else {
      parentFirst(parserManager, parent)
    }
  }
}