package org.mule.weave.v2.interpreted

import org.mule.weave.v2.core.util.ComposedMap
import org.mule.weave.v2.interpreted.debugger.server.DefaultWeaveDebuggingSession
import org.mule.weave.v2.interpreted.debugger.server.ServerProtocol
import org.mule.weave.v2.interpreted.debugger.server.WeaveDebuggerExecutor
import org.mule.weave.v2.interpreted.debugger.server.WeaveDebuggingSession
import org.mule.weave.v2.interpreted.listener.NotificationManager
import org.mule.weave.v2.interpreted.listener.WatchdogExecutionListener
import org.mule.weave.v2.interpreted.listener.WeaveExecutionListener
import org.mule.weave.v2.interpreted.module.WeaveDataFormat
import org.mule.weave.v2.interpreted.node.AstWrapperRoot
import org.mule.weave.v2.interpreted.node.structure.DocumentNode
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBindings
import org.mule.weave.v2.interpreted.node.structure.header.directives.DirectiveOption
import org.mule.weave.v2.interpreted.node.structure.header.directives.OutputDirective
import org.mule.weave.v2.interpreted.transform.EngineGrammarTransformation
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.writer.ConfigurableEncoding
import org.mule.weave.v2.module.writer.EmptyWriter
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.module.writer.WriterHelper
import org.mule.weave.v2.parser.ast.structure.{ DocumentNode => AstDocumentNode }
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.CompilationPhase
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware
import org.mule.weave.v2.parser.phase.SuccessResult
import org.mule.weave.v2.runtime.CompilationResult
import org.mule.weave.v2.runtime.DebugAwareWeave
import org.mule.weave.v2.runtime.ExecutableWeave

import java.io.ByteArrayOutputStream
import java.nio.charset.Charset
import scala.collection.mutable.ArrayBuffer

class InterpreterMappingCompilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler) extends CompilationPhase[AstNodeResultAware[AstDocumentNode] with ScopeNavigatorResultAware, CompilationResult[AstDocumentNode]] {

  override def doCall(source: AstNodeResultAware[AstDocumentNode] with ScopeNavigatorResultAware, context: ParsingContext): PhaseResult[CompilationResult[AstDocumentNode]] = {
    val astNode: AstDocumentNode = source.astNode
    val transform: DocumentNode = EngineGrammarTransformation(context, source.scope.astNavigator(), moduleNodeLoader).transformDocument(astNode)
    SuccessResult(CompilationResult[AstDocumentNode](astNode, new InterpretedMappingExecutableWeave(transform, astNode), source.scope), context)
  }
}

class InterpretedMappingExecutableWeave(val executableDocument: DocumentNode, val astDocument: AstDocumentNode) extends ExecutableWeave[AstDocumentNode] with DebugAwareWeave {

  private val nodeListeners: ArrayBuffer[WeaveExecutionListener] = ArrayBuffer()
  private var materializeValues: Boolean = false
  private var maxExecutionTime: Long = -1
  private val readerOptions = executableDocument.header.externalBindings
    .variables()
    .map((id) => {
      (id.name.name, id.options.getOrElse(Seq()))
    })
    .toMap

  private val runtimeOutputDirective = executableDocument.header.directives.collectFirst({ case od: OutputDirective => od })

  private val writerOptions: Seq[DirectiveOption] =
    runtimeOutputDirective
      .map((od) => {
        od.options.getOrElse(Seq())
      })
      .getOrElse(Seq())

  /**
    * Dumps the execution tree
    *
    * @return The result
    */
  def dumpExecutionTree(): (Any, Charset) = {
    implicit val ctx: EvaluationContext = EvaluationContext()
    val value: DataFormat[_, _] = DataFormatManager.byContentType("application/xml").getOrElse(new WeaveDataFormat())
    val writer = value.writer(Some(new ByteArrayOutputStream()))
    val context: ExecutionContext = ExecutionContext(
      writer = writer,
      variableTable = variableTable(),
      moduleTable = moduleTable(),
      notificationManager = createNotificationManager(),
      materializedValue = materializeValues,
      location = astDocument)
    try {
      writer.startDocument(executableDocument)
      writer.writeValue(new AstWrapperRoot(executableDocument))(context)
      writer.endDocument(executableDocument.body)
    } finally {
      context.close()
    }
    (writer.result, writer.settings match {
      case e: ConfigurableEncoding => e.charset
      case _                       => context.serviceManager.charsetProviderService.defaultCharset()
    })
  }

  /**
    * Registers an execution listener
    *
    * @param listener The listener
    */
  def addExecutionListener(listener: WeaveExecutionListener): Unit = {
    nodeListeners += listener
  }

  /**
    * The max time in milliseconds that this script can take. If it takes more time an exception will be thrown
    *
    * @param maxExecutionTime The max time in ms
    * @return
    */
  def withMaxTime(maxExecutionTime: Long): InterpretedMappingExecutableWeave = {
    this.maxExecutionTime = maxExecutionTime
    this
  }

  /**
    * Enables debugging over this weave script
    *
    * @param debuggerProtocol The debugger protocol
    * @return The session
    */
  override def debug(debuggerProtocol: ServerProtocol): DefaultWeaveDebuggingSession = {
    val session: DefaultWeaveDebuggingSession = new DefaultWeaveDebuggingSession(debuggerProtocol)
    debug(session)
    session
  }

  private def debug(session: WeaveDebuggingSession): Unit = {
    val debuggerExecutor: WeaveDebuggerExecutor = new WeaveDebuggerExecutor(session)
    session.start(debuggerExecutor)
    addExecutionListener(debuggerExecutor)
    materializeValues = true
  }

  def materializedValuesExecution(materialize: Boolean): Unit = {
    materializeValues = materialize
  }

  private def variableTable() = {
    executableDocument._1.variableTable
  }

  private def moduleTable() = {
    executableDocument._1.moduleTable
  }

  private def externalBindings(): ExternalBindings = {
    executableDocument._1.externalBindings
  }

  override def execute(readers: Map[String, Reader], values: Map[String, Value[_]])(implicit ctx: EvaluationContext): Value[_] = {
    val readerValues = readers.map((reader) => {
      (reader._1, new ReaderValue(reader._2, reader._1, readerOptions.getOrElse(reader._1, Seq())))
    })
    val notificationManager = createNotificationManager()
    val context: ExecutionContext = ExecutionContext(
      variableTable = variableTable(),
      moduleTable = moduleTable(),
      externalBindings = externalBindings(),
      values = ComposedMap(readerValues, values),
      notificationManager = notificationManager,
      materializedValue = materializeValues,
      evaluationContext = ctx,
      location = astDocument,
      resource = astDocument.location().resourceName)
    try {
      onExecutionStarted(notificationManager)
      if (notificationManager.nonEmpty) {
        ctx.registerCloseable(new AutoCloseable() {
          override def close(): Unit = {
            //Call the execution ended when the context is closed
            onExecutionEnded(notificationManager)
          }
        })
      }
      executableDocument.execute(context)
    } catch {
      case e: Exception => {
        if (ctx.serviceManager.settingsService.dumper().enabled) {
          if (values.contains("app")) {
            val withoutRegistry: Seq[KeyValuePair] = values("app").evaluate.asInstanceOf[ObjectSeq].toSeq().filter((kv: KeyValuePair) => kv._1.evaluate.name != "registry")
            val valuesWithoutRegistry = values + ("app" -> ObjectValue(ObjectSeq(withoutRegistry)))
            Dumper(valuesWithoutRegistry ++ readerValues, e, EmptyWriter, executableDocument, astDocument).dumpFilesToReproduceError()
          } else {
            Dumper(values ++ readerValues, e, EmptyWriter, executableDocument, astDocument).dumpFilesToReproduceError()
          }
        }
        throw e
      }
    }
  }

  private def onExecutionEnded(notificationManager: NotificationManager)(implicit ctx: EvaluationContext): Unit = {
    try {
      notificationManager.onExecutionEnded(this)
    } catch {
      case e: Exception => {
        ctx.serviceManager.loggingService.logError("Internal error when executing `onExecutionEnded` notification manager: " + e.getMessage)
      }
    }
  }

  private def onExecutionStarted(notificationManager: NotificationManager)(implicit ctx: EvaluationContext): Unit = {
    try {
      notificationManager.onExecutionStarted(this)
    } catch {
      case e: Exception => {
        ctx.serviceManager.loggingService.logError("Internal error when executing `onExecutionStarted` notification manager: " + e.getMessage)
      }
    }
  }

  private def createNotificationManager(): NotificationManager = {
    if (maxExecutionTime > -1) {
      NotificationManager((nodeListeners :+ new WatchdogExecutionListener(maxExecutionTime)).toArray)
    } else if (nodeListeners.isEmpty) {
      NotificationManager.EMPTY
    } else {
      NotificationManager(nodeListeners.toArray)
    }
  }

  override def writeWith(writer: Writer, readers: Map[String, Reader], values: Map[String, Value[_]], closeAfterWrite: Boolean)(implicit ctx: EvaluationContext): (Any, Charset) = {
    val readerValues = readers.map((reader) => {
      (reader._1, new ReaderValue(reader._2, reader._1, readerOptions.getOrElse(reader._1, Seq())))
    })
    val notificationManager = createNotificationManager()
    val context: ExecutionContext = ExecutionContext(
      writer = writer,
      variableTable = variableTable(),
      moduleTable = moduleTable(),
      externalBindings = externalBindings(),
      values = ComposedMap(readerValues, values),
      notificationManager = notificationManager,
      evaluationContext = ctx,
      materializedValue = materializeValues,
      location = astDocument)
    try {
      onExecutionStarted(notificationManager)
      if (notificationManager.nonEmpty) {
        ctx.registerCloseable(new AutoCloseable() {
          override def close(): Unit = {
            //Call the execution ended when the context is closed
            onExecutionEnded(notificationManager)
          }
        })
      }
      WriterHelper.writeAndGetResult(writer, executableDocument.execute(context), executableDocument)(context)
    } catch {
      case e: Exception => {
        if (ctx.serviceManager.settingsService.dumper().enabled) {
          if (values.contains("app")) {
            val withoutRegistry = values("app").evaluate.asInstanceOf[ObjectSeq].toSeq().filter((kv: KeyValuePair) => kv._1.evaluate.name != "registry")
            val valuesWithoutRegistry = values + ("app" -> ObjectValue(ObjectSeq(withoutRegistry)))
            Dumper(valuesWithoutRegistry ++ readerValues, e, writer, executableDocument, astDocument).dumpFilesToReproduceError()
          } else {
            Dumper(values ++ readerValues, e, writer, executableDocument, astDocument).dumpFilesToReproduceError()
          }
        }
        throw e
      }
    } finally {
      if (closeAfterWrite) {
        context.close()
      }

    }
  }

  override def configureWriter(writer: Writer)(implicit ctx: EvaluationContext): Writer = {
    writerOptions.foreach((option) => {
      ConfigurationHelper.configure(writer, option)
    })
    writer
  }

  override def removeExecutionListener(listener: WeaveExecutionListener): Unit = {
    nodeListeners -= listener
  }
}
