package org.mule.weave.v2.interpreted.node.executors

import org.mule.weave.v2.core.functions.BaseUnaryFunctionValue
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.UnexpectedFunctionCallTypesException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.allTargets
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.findMatchingFunctionWithCoercion
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.indexOfFunction
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.materializeOverloadedFunctionArgs
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.structure.ConstantValueNode
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValuesHelper
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.runtime.core.functions.OverloadedFunctionValue

import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.atomic.AtomicReference

abstract class UnaryOverloadedStaticExecutor(val isOperator: Boolean) extends UnaryExecutor {

  private val cachedDispatchIndex: AtomicInteger = new AtomicInteger(-1)
  private val cachedCoercedDispatchIndex: AtomicReference[(Int, Type)] = new AtomicReference()

  override def execute(arguments: Array[Value[_]])(implicit ctx: ExecutionContext): Value[Any] = {
    //This should always be like this as they are operators
    executeUnary(arguments(0))
  }

  def executeUnary(fv: Value[_])(implicit ctx: ExecutionContext): Value[Any] = {
    val targetFunction: FunctionValue = target()
    val resolvedTargets: Array[_ <: FunctionValue] = allTargets(targetFunction)

    val dispatchIndex = cachedDispatchIndex.get()
    if (dispatchIndex != -1) {
      val operation: BaseUnaryFunctionValue = resolvedTargets(dispatchIndex).asInstanceOf[BaseUnaryFunctionValue]
      //If values are literal we do not need to validate every just the first time when we load the operation
      //Then is ok every time as its type will NEVER change
      val firstValue = resolveCachedDispatchValue(fv, operation)

      if (firstArgConstantType || operation.R.accepts(firstValue)) {
        return doCall(firstValue, operation)
      }
    }

    val firstValue: Value[_] = if (targetFunction.paramsTypesRequiresMaterialize) {
      materializeOverloadedFunctionArgs(resolvedTargets, 0, fv)
    } else {
      fv
    }

    val coercedOperation: (Int, Type) = cachedCoercedDispatchIndex.get
    if (coercedOperation != null) {
      //If values are literal we do not need to validate every just the first time when we load the coerced operation
      //Then is ok every time as its type will NEVER change
      if (firstArgConstantType || coercedOperation._2.accepts(firstValue)) {
        val functionToDispatch = resolvedTargets(coercedOperation._1).asInstanceOf[BaseUnaryFunctionValue]
        val mayBeFirstValue = functionToDispatch.R.coerceMaybe(firstValue)
        if (mayBeFirstValue.isDefined) {
          return doCall(mayBeFirstValue.get, functionToDispatch)
        }
      }
    }

    val matchingOp: Int = findMatchingFunction(resolvedTargets, firstValue)
    if (matchingOp != -1) {
      //Update it here should we use any strategy
      if (targetFunction.dispatchCanBeCached) {
        cachedDispatchIndex.set(matchingOp)
      }
      doCall(firstValue, resolvedTargets(matchingOp).asInstanceOf[BaseUnaryFunctionValue])
    } else {
      val materializedValues: Array[Value[Any]] = ValuesHelper.array(firstValue.materialize)
      val argTypes: Array[Type] = materializedValues.map(_.valueType)
      val sortedOperators: Array[FunctionValue] = FunctionDispatchingHelper.sortByParameterTypeWeight(resolvedTargets, argTypes)
      val functionToCallWithCoercion: Option[(Int, Array[Value[_]], Seq[Int])] = findMatchingFunctionWithCoercion(materializedValues, sortedOperators, this)
      functionToCallWithCoercion match {
        case Some((functionToDispatch, argumentsWithCoercion, _)) => {
          //Cache the coercion use the base type to avoid Memory Leaks as Types may have references to Streams or Objects
          val binaryFunctionValue = sortedOperators(functionToDispatch).asInstanceOf[BaseUnaryFunctionValue]
          if (targetFunction.dispatchCanBeCached) {
            cachedCoercedDispatchIndex.set((indexOfFunction(resolvedTargets, binaryFunctionValue), firstValue.valueType.baseType))
          }
          val firstCoercedValue: Value[_] = argumentsWithCoercion(0)
          doCall(firstCoercedValue, binaryFunctionValue)
        }
        case None =>
          throw new UnexpectedFunctionCallTypesException(location(), name, materializedValues, sortedOperators.map(_.parameters.map(_.wtype)))
      }
    }
  }

  final def resolveCachedDispatchValue(fv: Value[_], operation: BaseUnaryFunctionValue)(implicit ctx: ExecutionContext): Value[_] = {
    if (isOperator) {
      fv
    } else {
      if (!firstArgConstantType && operation.rightParam.typeRequiresMaterialization) {
        fv.materialize
      } else {
        fv
      }
    }
  }

  final def doCall(leftValue: Value[_], operation: BaseUnaryFunctionValue)(implicit ctx: ExecutionContext): Value[_] = {
    if (isOperator) {
      operation.call(leftValue)
    } else {
      try {
        operation.call(leftValue)
      } catch {
        case ex: ExecutionException =>
          if (showInStacktrace()) {
            ex.addCallToStacktrace(location(), name())
          }
          throw ex
      }
    }
  }

  def findMatchingFunction(operations: Array[_ <: FunctionValue], firstValue: Value[_])(implicit ctx: ExecutionContext): Int = {
    var i = 0
    while (i < operations.length) {
      val operation: BaseUnaryFunctionValue = operations(i).asInstanceOf[BaseUnaryFunctionValue]
      if (operation.R.accepts(firstValue)) {
        return i
      }
      i = i + 1
    }
    -1
  }

  def firstArgConstantType: Boolean

  def target()(implicit ctx: ExecutionContext): FunctionValue

}

class UnaryOpExecutor(targetOperators: Array[_ <: BaseUnaryFunctionValue], val name: String, override val location: WeaveLocation, val firstArgConstantType: Boolean)
    extends UnaryOverloadedStaticExecutor(isOperator = true) with Product3[Array[_ <: BaseUnaryFunctionValue], String, Boolean] {

  private val function = OverloadedFunctionValue.createValue(targetOperators, Array.empty, Some(this.name), location, cacheabe = true)

  override def target()(implicit ctx: ExecutionContext): FunctionValue = {
    function
  }

  override def name()(implicit ctx: ExecutionContext): String = this.name

  override def showInStacktrace(): Boolean = false

  override def _1: Array[_ <: BaseUnaryFunctionValue] = targetOperators

  override def _2: String = name

  override def _3: Boolean = firstArgConstantType

  override def node(): ValueNode[_] = {
    ConstantValueNode(function)
  }
}

/**
  * This class has the logic to execute unary functions. It knows how to dispatch how to do coercions and such
  */
class UnaryOverloadedFunctionExecutor(
  val node: ValueNode[_],
  val name: String,
  val firstArgConstantType: Boolean,
  val showInStacktrace: Boolean = true,
  override val location: WeaveLocation) extends UnaryOverloadedStaticExecutor(isOperator = false) with Product3[ValueNode[_], String, Boolean] {

  override def target()(implicit ctx: ExecutionContext): FunctionValue = {
    node.execute.asInstanceOf[FunctionValue]
  }

  override def executeUnary(firstArgument: Value[_])(implicit ctx: ExecutionContext): Value[Any] = {
    val activeFrame: Frame = ctx.executionStack().activeFrame()
    try {
      activeFrame.updateCallSite(node)
      super.executeUnary(firstArgument)
    } finally {
      activeFrame.cleanCallSite()
    }
  }

  override def name()(implicit ctx: ExecutionContext): String = this.name

  override def _1: ValueNode[_] = node

  override def _2: String = name

  override def _3: Boolean = firstArgConstantType
}